 /*
  * <LIC_AMD_STD>
  * Copyright (C) <years> Advanced Micro Devices, Inc.  All Rights Reserved.
  * </LIC_AMD_STD>
  * 
  * <CTL_AMD_STD>
  * </CTL_AMD_STD>
  * 
  * <DOC_AMD_STD>
  * Miniport entry, power management code and IOCTL code.
  * </DOC_AMD_STD>
  * 
  */

#include "dderror.h"
#include "devioctl.h"
#include "miniport.h"

/* DPMS SUPPORT FOR WIN2K AND AFTER */

#if _WIN32_WINNT >= 0x0500
#define WIN2K_DPMS 1
#include "ntddvdeo.h"
#endif 

#include "video.h"
#include "gxdrv.h"
#include "debug.h"
#include "gfx_rtns.h"
#include "gxinfo.h"
#include "escape.h"
#include "drv_rtns.h"
#include "gfx_defs.h"
#include "gfx_regs.h"

/* CHILD DEVICE DEFINITIONS */

#define P2_DDC_MONITOR        (0x1357bd00)
#define P2_NONDDC_MONITOR     (0x1357bd01)

#define DRIVER_FLAG_LINEAR_MODE 0x80000000l
#define DRIVER_FLAG_COMPRESSION 0x40000000l

extern void gfx_outw (unsigned short port, unsigned short address);
extern unsigned short gfx_inw (unsigned short port);

ULONG *fb_ptr = NULL;
ULONG heapStart     = 0;
ULONG heapEnd       = 0;
int from_hibernate  = 0;

/* VARIABLES!!! */
/* The current (shoddy?) design of using Durango only in the miniport */
/* leaves much to be desired.  The display driver needs crucial       */
/* information, such as the panel resolution and the current panning  */
/* position.  We must therefore (temporarily) expose these variables  */
/* through the durango functions.                                     */

extern unsigned short PanelWidth;
extern unsigned short PanelHeight;
extern unsigned short PanelEnable;

extern int DeltaX;
extern int DeltaY;

VIDEO_ACCESS_RANGE GXLegacyResourceList[] = 
{
    {0x000C0000, 0x00000000, 0x00010000, 0, 0, 0, 0}, // ROM location
    {0x000A0000, 0x00000000, 0x00020000, 0, 0, 1, 0}, // Frame buffer
    {0x000003B0, 0x00000000, 0x0000000B, 1, 1, 1, 0}, // VGA regs
    {0x000003C0, 0x00000000, 0x00000020, 1, 1, 1, 0}  // VGA regs
};
ULONG GXLegacyResourceEntries = sizeof GXLegacyResourceList / sizeof GXLegacyResourceList[0];

/*-------------------------------------------------------------------------
 * DriverEntry
 *
 * This is the first miniport routine called by the kernel.
 * We aren't supposed to understand Context1 and Context2. we just pass 
 * them to VideoPortInitialize.
 *
 * Routine Description:
 *
 *  Installable driver initialization entry point.
 *  This entry point is called directly by the I/O system.
 *
 * Arguments:
 *
 *  Context1 - First context value passed by the operating system. This is
 *             the value with which the miniport driver calls VideoPortInitialize().
 *
 *  Context2 - Second context value passed by the operating system. This is
 *             the value with which the miniport driver calls VideoPortInitialize().
 *
 * Return Value:
 *
 *  Status from VideoPortInitialize()
 -------------------------------------------------------------------------*/

ULONG DriverEntry(PVOID Context1, PVOID Context2)
{
	VIDEO_HW_INITIALIZATION_DATA hwInitData;
	ULONG initStatus;

	DISPDBG((1, "\n\n\nStarting Driver\nDriverEntry ----------\n"));

	VideoPortZeroMemory(&hwInitData, sizeof(VIDEO_HW_INITIALIZATION_DATA));

	initStatus = ERROR_DEV_NOT_EXIST;

	/* INITIALIZE THE STRUCTURE */
	/* At this stage, we fill in an initialization structure with important information, */
	/* such as pointers to other routines that can be called.                            */

	hwInitData.HwInitDataSize            = sizeof(VIDEO_HW_INITIALIZATION_DATA);
	hwInitData.HwFindAdapter             = GXFindAdapter;
	hwInitData.HwInitialize              = GXInitialize;
	hwInitData.HwResetHw                 = NULL;
	hwInitData.HwInterrupt               = NULL;
	hwInitData.HwStartIO                 = GXStartIO;
	hwInitData.HwDeviceExtensionSize     = sizeof(HW_DEVICE_EXTENSION);
	hwInitData.AdapterInterfaceType      = PCIBus;
#if _WIN32_WINNT >= 0x0500
#if WIN2K_DPMS
	hwInitData.HwSetPowerState           = GXSetPowerState;
	hwInitData.HwGetPowerState           = GXGetPowerState;
	hwInitData.HwGetVideoChildDescriptor = GXGetChildDescriptor;
#endif 
#endif 

	hwInitData.HwLegacyResourceList      = GXLegacyResourceList;
    hwInitData.HwLegacyResourceCount     = GXLegacyResourceEntries;

	initStatus = VideoPortInitialize(Context1, Context2, &hwInitData, NULL);

	DISPDBG((1, "DriverEntry: status from VideoPortInitialize %d\n", initStatus));
	DISPDBG((1, "DriverEntry returning...\n"));

	return initStatus;

}/* DriverEntry */

/*-------------------------------------------------------------------------
 * GeodeRegistryCallback
 *
 * This routine is called by the videoport driver to parse a value extracted from 
 * the registry.  The address of this routine is supplied when requesting an extraction.
 -------------------------------------------------------------------------*/

VP_STATUS GeodeRegistryCallback(
    PVOID HwDeviceExtension,
    PVOID Context,
    PWSTR ValueName,
    PVOID ValueData,
    ULONG ValueLength
    )
{
    if (ValueLength) 
    {
        if (Context) 
        {                  
            *(ULONG *)Context = *(PULONG)ValueData;
        }
        else if (*((PULONG)ValueData) != 0)
        {                  
            return ERROR_INVALID_PARAMETER;
        }

        return NO_ERROR;
    } 
	else 
    {
        return ERROR_INVALID_PARAMETER;
    }
} 

/*-------------------------------------------------------------------------
 * GXFindAdapter
 *
 * Make sure we're running a GX and that video memory has been allocated.
 *
 * Routine Description:
 *
 *  This routine is called to determine if the adapter for this driver
 *  is present in the system.
 *  If it is present, the function fills out some information describing
 *  the adapter.
 *
 * Arguments:
 *
 *  HwDeviceExtension - Supplies the miniport driver's adapter storage. This
 *                      storage is initialized to zero before this call.
 *
 *  HwContext         - Supplies the context value which was passed to
 *                      VideoPortInitialize().
 *
 *  ArgumentString    - Supplies a NULL terminated ASCII string. This string
 *                      originates from the user.
 *
 *  ConfigInfo        - Returns the configuration information structure which is
 *                      filled by the miniport driver. This structure is initialized with
 *                      any knwon configuration information (such as SystemIoBusNumber) by
 *                      the port driver. Where possible, drivers should have one set of
 *                      defaults which do not require any supplied configuration information.
 *
 *  Again             - Indicates if the miniport driver wants the port driver to call
 *                      its VIDEO_HW_FIND_ADAPTER function again with a new device extension
 *                      and the same config info. This is used by the miniport drivers which
 *                      can search for several adapters on a bus.
 *
 * Return Value:
 *
 *  This routine must return:
 *
 *    NO_ERROR                - Indicates a host adapter was found and the
 *                              configuration information was successfully determined.
 *
 *    ERROR_INVALID_PARAMETER - Indicates an adapter was found but there was an
 *                              error obtaining the configuration information. If possible an error
 *                              should be logged.
 *
 *    ERROR_DEV_NOT_EXIST     - Indicates no host adapter was found for the
 *                              supplied configuration information.
 -------------------------------------------------------------------------*/

VP_STATUS GXFindAdapter(
	PHW_DEVICE_EXTENSION HwDeviceExtension,
	PVOID HwContext,
	PWSTR ArgumentString,
	PVIDEO_PORT_CONFIG_INFO ConfigInfo,
	PUCHAR Again)
{
	VIDEO_ACCESS_RANGE accessRanges[8];
	VP_STATUS status = 0;
	ULONG RegBase    = 0;
	ULONG VidBase    = 0;
	ULONG GPBase     = 0;
	ULONG FBBase     = 0;
	ULONG FBSize     = 0;
    ULONG counter    = 0;
	ULONG ulValue;
	
	DISPDBG((1, "GXFindAdapter: enter\n"));

	/* CLEAR RECURSIVE FLAG */
	/* We don't want to be called again */
	
	*Again = 0; 	

	/* INITIALIZE DURANGO */

	gfx_detect_cpu();

	/* NT 4 COMPATIBILITY */
	/* To be binary compatible with WinNT 4, some routines are extracted runtime, instead of */
	/* being linked to statically.                                                          */

#if _WIN32_WINNT >= 0x0500
        
	if(!(HwDeviceExtension->Win2kVideoPortGetRomImage =  
              ConfigInfo->VideoPortGetProcAddress( HwDeviceExtension, 
                                                    "VideoPortGetRomImage")))
    {
        return (ERROR_DEV_NOT_EXIST);
    }

    if(!(HwDeviceExtension->Win2kVideoPortGetCommonBuffer = 
         ConfigInfo->VideoPortGetProcAddress( HwDeviceExtension, 
                                             "VideoPortGetCommonBuffer")))
    {
        return (ERROR_DEV_NOT_EXIST);
    }
    if(!(HwDeviceExtension->Win2kVideoPortFreeCommonBuffer =
         ConfigInfo->VideoPortGetProcAddress( HwDeviceExtension, 
                                             "VideoPortFreeCommonBuffer")))
    {
        return (ERROR_DEV_NOT_EXIST);
    }
    if(!(HwDeviceExtension->Win2kVideoPortDDCMonitorHelper =
         ConfigInfo->VideoPortGetProcAddress( HwDeviceExtension, 
                                             "VideoPortDDCMonitorHelper")))
    {
        return (ERROR_DEV_NOT_EXIST);
    }
    if(!(HwDeviceExtension->Win2kVideoPortInterlockedExchange =
          ConfigInfo->VideoPortGetProcAddress( HwDeviceExtension, 
                                              "VideoPortInterlockedExchange")))
    {
        return (ERROR_DEV_NOT_EXIST);
    }
    if(!(HwDeviceExtension->Win2kVideoPortGetVgaStatus =
          ConfigInfo->VideoPortGetProcAddress( HwDeviceExtension, 
                                              "VideoPortGetVgaStatus")))
    {
        return (ERROR_DEV_NOT_EXIST);
    }

#endif /* _WIN32_WINNT >= 0x500 */

    if (!gfx_cpu_version)
	{
		DISPDBG((99999, "Not a AMD Geode part\n"));
		return(ERROR_DEV_NOT_EXIST);
	}

	/* RETRIEVE BASE ADDRESSES         */
	/* Different for each architecture */

	RegBase  = gfx_get_cpu_register_base();
	VidBase  = gfx_get_vid_register_base();
	FBBase   = gfx_get_frame_buffer_base();
	FBSize   = gfx_get_frame_buffer_size();
	GPBase   = gfx_get_graphics_register_base();

	/* ADD RANGES TO ACCESS RANGE ARRAY                           */
	/* The macro takes care of checking if the range is relevant. */

	VideoPortZeroMemory (accessRanges, sizeof(accessRanges));
	ADD_TO_ACCESS_RANGES (counter, RegBase, 0x1000,  0);
	ADD_TO_ACCESS_RANGES (counter, VidBase, 0x1000,  0);
	ADD_TO_ACCESS_RANGES (counter, FBBase,  FBSize,  0);
	ADD_TO_ACCESS_RANGES (counter, GPBase,  0x1000,  0);
	ADD_TO_ACCESS_RANGES (counter, 0x3C0,   31,      1);
		
	/* CLEAR OUT EMULATOR ENTRIES AND STATE SIZE */
	/* This driver does not support them.        */

	ConfigInfo->NumEmulatorAccessEntries     = 0;
	ConfigInfo->EmulatorAccessEntries        = NULL;
	ConfigInfo->EmulatorAccessEntriesContext = 0;

	ConfigInfo->HardwareStateSize = 0;

	ConfigInfo->VdmPhysicalVideoMemoryAddress.LowPart  = 0x00000000;
	ConfigInfo->VdmPhysicalVideoMemoryAddress.HighPart = 0x00000000;
	ConfigInfo->VdmPhysicalVideoMemoryLength           = 0x00000000;

	/* MAP ADDRESSES IN DEVICE MEMORY */

	if (GXMapPhysicalAddress (HwDeviceExtension, RegBase, 0x1000,  FALSE, (PULONG)&gfx_virt_regptr))
		return(ERROR_DEV_NOT_EXIST);

	if (GXMapPhysicalAddress (HwDeviceExtension, VidBase, 0x1000,  FALSE, (PULONG)&gfx_virt_vidptr))
		return(ERROR_DEV_NOT_EXIST);

	if (GXMapPhysicalAddress (HwDeviceExtension, FBBase,  FBSize,  FALSE, (PULONG)&gfx_virt_fbptr))
		return(ERROR_DEV_NOT_EXIST);

	if (GXMapPhysicalAddress (HwDeviceExtension, GPBase,  0x1000,  FALSE, (PULONG)&gfx_virt_gpptr))
		return(ERROR_DEV_NOT_EXIST);

	if (GXMapPhysicalAddress (HwDeviceExtension, 0x3C0,   31,      TRUE,  (PULONG)&(HwDeviceExtension->GXVGAAddr)))
		return(ERROR_DEV_NOT_EXIST);

	/* STORE PHYSICAL ADDRESSES IN DEVICE MEMORY FOR LATER */

	HwDeviceExtension->PhysGXRegAddr.LowPart  = RegBase;
	HwDeviceExtension->PhysGXRegAddr.HighPart = 0;
	HwDeviceExtension->PhysGXVidAddr.LowPart  = VidBase;
	HwDeviceExtension->PhysGXVidAddr.HighPart = 0;
	HwDeviceExtension->PhysGXVGAAddr.LowPart  = 0x3C0;
	HwDeviceExtension->PhysGXVGAAddr.HighPart = 0;
	HwDeviceExtension->PhysGXFBAddr.LowPart   = FBBase;
	HwDeviceExtension->PhysGXFBAddr.HighPart  = 0;
	HwDeviceExtension->PhysGXGPAddr.LowPart   = GPBase;
	HwDeviceExtension->PhysGXGPAddr.HighPart  = 0;
	HwDeviceExtension->FBSize                 = FBSize;

	/* SET REGISTRY PARAMETERS */

	VideoPortSetRegistryParameters(HwDeviceExtension,
								   L"HardwareInformation.ChipType",
								   GX_CHIP_TYPE,
								   sizeof(GX_CHIP_TYPE));

	VideoPortSetRegistryParameters(HwDeviceExtension,
								   L"HardwareInformation.DacType",
								   GX_DAC,
								   sizeof(GX_DAC));

	VideoPortSetRegistryParameters(HwDeviceExtension,
								   L"HardwareInformation.MemorySize",
								   &HwDeviceExtension->FBSize,
								   sizeof(ULONG));

	VideoPortSetRegistryParameters(HwDeviceExtension,
								   L"HardwareInformation.AdapterString",
								   GX_ADAPTER,
								   sizeof(GX_ADAPTER));

	/* RETRIEVE USER (OR APPLICATION) SUPPLIED PARAMETERS */

	/* COMPRESSION */
	/* 0 = Compression Off                   */
	/* 1 = Compression On when enough memory */
	/* 2 = Compression always on.            */

    if (VideoPortGetRegistryParameters(HwDeviceExtension,
                                       L"Compression",
                                       FALSE,
                                       GeodeRegistryCallback,
                                       &ulValue) == NO_ERROR) 
	{
		HwDeviceExtension->Compression = ulValue;
	} 
	else 
	{
		HwDeviceExtension->Compression = 0;
	}

	DISPDBG((2000, "Compression - %d\n", HwDeviceExtension->Compression));

	/* LINEAR FRAME BUFFER                      */
	/* 0 = Power of 2 line pitch.               */
	/* 1 = Power of 2 line pitch when possible. */
	/* 2 = Linear Frame buffer                  */
	
    if (VideoPortGetRegistryParameters(HwDeviceExtension,
                                       L"FBLinear",
                                       FALSE,
                                       GeodeRegistryCallback,
                                       &ulValue) == NO_ERROR) 
	{
		HwDeviceExtension->LinearFB = ulValue;
	} 
	else 
	{
		HwDeviceExtension->LinearFB = 0;
	}

	DISPDBG((2000, "Linear Frame Buffer - %d\n", HwDeviceExtension->LinearFB));

	/* DDC SUPPORT */

    if (VideoPortGetRegistryParameters(HwDeviceExtension,
                                       L"DDCEnabled",
                                       FALSE,
                                       GeodeRegistryCallback,
                                       &ulValue) == NO_ERROR) 
	{
		HwDeviceExtension->DDCEnabled = ulValue;
	} 
	else 
	{
		HwDeviceExtension->DDCEnabled = 0;
	}

	/* ANTIALIASED TEXT SUPPORT */

    if (VideoPortGetRegistryParameters(HwDeviceExtension,
                                       L"AntialiasedText",
                                       FALSE,
                                       GeodeRegistryCallback,
                                       &ulValue) == NO_ERROR) 
	{
		HwDeviceExtension->AntialiasedText = ulValue;
	} 
	else 
	{
		HwDeviceExtension->AntialiasedText = 0;
	}

	DISPDBG((2000, "AntialiasedText - %d\n", HwDeviceExtension->AntialiasedText));

	/* READ PANEL SETTINGS */
	/* GX2 is bonded out for either TFT or CRT.  There is no */
	/* reason therefore to ever disable panel support.       */
			
	gfx_outw (VR_INDEX, VR_UNLOCK);
	gfx_outw (VR_INDEX, (VRC_VG << 8) | VG_MEM_SIZE);
	
	if (gfx_inw (VR_DATA) & FP_DETECT_MASK)
	{
		/* DISABLE DDC FOR PANEL OUTPUT */

		HwDeviceExtension->DDCEnabled = 0;

		gfx_outw (VR_INDEX, VR_UNLOCK);
		gfx_outw (VR_INDEX, (VRC_VG << 8) | VG_FP_TYPE);
		
		switch (gfx_inw (VR_DATA) & FP_RESOLUTION_MASK)
		{
			case FP_RES_640_480:
				HwDeviceExtension->PanelWidth  = 640;
				HwDeviceExtension->PanelHeight = 480;
				break;
			
			case FP_RES_800_600:
				HwDeviceExtension->PanelWidth  = 800;
				HwDeviceExtension->PanelHeight = 600;
				break;
			
			case FP_RES_1024_768:
				HwDeviceExtension->PanelWidth  = 1024;
				HwDeviceExtension->PanelHeight = 768;
				break;
			
			case FP_RES_1280_1024:
				HwDeviceExtension->PanelWidth  = 1280;
				HwDeviceExtension->PanelHeight = 1024;
				break;
		}
	}
	else
	{
		HwDeviceExtension->PanelWidth  = 0;
		HwDeviceExtension->PanelHeight = 0;
	}
	
	/* FILTER VIDEO MODES THAT ARE NOT SUPPORTED BY THE MONITOR */

	if (!FilterUnsupportedModes(HwDeviceExtension))
		return(ERROR_DEV_NOT_EXIST);

	/* INDICATE SUCCESSFUL COMPLETION */
	
	return (NO_ERROR);

} /* GXFindAdapter */

/*-------------------------------------------------------------------------
 * FilterUnsupportedModes
 *
 * Figure out which modes are supported by the current framebuffer model
 * and remove those we can't support. 
 -------------------------------------------------------------------------*/

BOOLEAN FilterUnsupportedModes (PHW_DEVICE_EXTENSION HwDeviceExtension)
{
	ULONG curModeIndex;
	ULONG maxStride, minStride;
	ULONG modeSize, compSize;
	ULONG fbmax;
	BOOLEAN reject;
	int supported = 0;
	int i;	
	
	/* CLEAR INCREMENTAL VARIABLES */

	curModeIndex = 0;	

	/* CALCULATE END OF FRAME BUFFER       */
	/* We always leave room for the cursor */

	fbmax = HwDeviceExtension->FBSize - 0x5000;
		
	for (i = 0; i < (int)gxVideoModeCount; i++) 
	{
		/* ASSUME WE ACCEPT THE MODE */

		reject = FALSE;

		/* FIRST FILTER THE MODE THROUGH DURANGO                     */
		/* Durango does not do any tests for a large enough frame    */
		/* buffer.  These are done manually.                         */

        if (HwDeviceExtension->PanelWidth)
        {
            if (gxVideoModes[i].ntVideoMode.Frequency != 60 ||
               (gfx_is_panel_mode_supported (
                    HwDeviceExtension->PanelWidth,
                    HwDeviceExtension->PanelHeight,
                    (unsigned short)gxVideoModes[i].ntVideoMode.VisScreenWidth,
                    (unsigned short)gxVideoModes[i].ntVideoMode.VisScreenHeight,
                    (unsigned short)gxVideoModes[i].ntVideoMode.BitsPerPlane) < 0))
            {
                reject = TRUE;
            }
        }
        else
        {
		    if (gfx_is_display_mode_supported(
			    gxVideoModes[i].ntVideoMode.VisScreenWidth,
			    gxVideoModes[i].ntVideoMode.VisScreenHeight,
			    gxVideoModes[i].ntVideoMode.BitsPerPlane,
			    gxVideoModes[i].ntVideoMode.Frequency) < 0)
		    {
			    reject = TRUE;
		    }
        }

		/* CALCULATE THE LINEAR AND RECTANGULAR STRIDES              */
		/* The rectangular stride is already in the video structure. */

		maxStride = gxVideoModes[i].ntVideoMode.ScreenStride;
		
		minStride = gxVideoModes[i].ntVideoMode.VisScreenWidth <<  
			        (gxVideoModes[i].ntVideoMode.BitsPerPlane >> 4);

		/* CALCULATE BASE MODE SIZE */
		/* If a linear frame buffer is disabled, we will always use */
		/* the rectangular stride.  For the other two options, the  */
		/* linear frame buffer will be available, thus reducing the */
		/* minimum mode footprint.                                  */

		if (HwDeviceExtension->LinearFB == 0)
		{
			modeSize = maxStride * gxVideoModes[i].ntVideoMode.VisScreenHeight;
		}
		else if (HwDeviceExtension->LinearFB == 1)
		{
			modeSize = maxStride * gxVideoModes[i].ntVideoMode.VisScreenHeight;

			/* RECTANGULAR IS TOO BIG - USE LINEAR */

			if (modeSize > fbmax)
			{
				modeSize = minStride * gxVideoModes[i].ntVideoMode.VisScreenHeight;
				gxVideoModes[i].ntVideoMode.DriverSpecificAttributeFlags |= DRIVER_FLAG_LINEAR_MODE;
				gxVideoModes[i].ntVideoMode.ScreenStride = minStride;
			}
		}
		else
		{
			modeSize = minStride * gxVideoModes[i].ntVideoMode.VisScreenHeight;
			gxVideoModes[i].ntVideoMode.DriverSpecificAttributeFlags |= DRIVER_FLAG_LINEAR_MODE;
			gxVideoModes[i].ntVideoMode.ScreenStride = minStride;
		}

        /* ADD COMPRESSION */
		/* Unless compression is requested to be always on, the minimum */
		/* mode footprint will have compression disabled.  Note that we */
        /* disable compression for linear modes, as frame-dirty mode    */
        /* is broken in the hardware in GX2                             */

        if ((gxVideoModes[i].ntVideoMode.ScreenStride == 1024 ||
             gxVideoModes[i].ntVideoMode.ScreenStride == 2048 ||
             gxVideoModes[i].ntVideoMode.ScreenStride == 4096 ||
             gxVideoModes[i].ntVideoMode.ScreenStride == 8192) &&
            HwDeviceExtension->Compression != 0)
		{
			if (HwDeviceExtension->PanelWidth && 
				(int)gxVideoModes[i].ntVideoMode.VisScreenHeight > HwDeviceExtension->PanelHeight)
			{
				compSize = 544 * HwDeviceExtension->PanelHeight;
			}
			else
			{
				compSize = 544 * gxVideoModes[i].ntVideoMode.VisScreenHeight;
			}			
            
			if (HwDeviceExtension->Compression == 2 || 
				(HwDeviceExtension->Compression == 1 && 
				((modeSize + compSize) < fbmax)))
			{
				modeSize += compSize;
				gxVideoModes[i].ntVideoMode.DriverSpecificAttributeFlags |= DRIVER_FLAG_COMPRESSION;
			}
		}

		if (modeSize > fbmax)
			reject = TRUE;
		
		/* SAVE THE OFFSCREEN END TO AVOID EXTRA WORK IN THE DISPLAY DRIVER */

		else
			gxVideoModes[i].ntVideoMode.DriverSpecificAttributeFlags |= modeSize;
		
		if(reject)
		{
			DISPDBG((10, "mode %d, %d bpp: (%d x %d @ %d) = %d bytes - rejected",
				i, gxVideoModes[i].ntVideoMode.BitsPerPlane,
				gxVideoModes[i].ntVideoMode.VisScreenWidth,
				gxVideoModes[i].ntVideoMode.VisScreenHeight, 
				gxVideoModes[i].ntVideoMode.Frequency, 
				modeSize));
		} 
		else 
		{
			/* COPY MODE INFORMATION DOWN TO KEEP LIST PACKED */
			
			if((unsigned int)i != curModeIndex) 
			{
				gxVideoModes[curModeIndex] = gxVideoModes[i];
			}

			/* SET THE INDEX OF THE MODE */
			/* Set the index of this mode. It has to be done   */
			/* somewhere and experience proves it's laborious  */
			/* and error prone to do it statically.            */

			gxVideoModes[curModeIndex].ntVideoMode.ModeIndex = curModeIndex;

			curModeIndex++;
			
			DISPDBG((10, "mode %d, %d bpp: (%d x %d @ %d) = %d bytes - accepted",
				i, gxVideoModes[i].ntVideoMode.BitsPerPlane,
				gxVideoModes[i].ntVideoMode.VisScreenWidth,
				gxVideoModes[i].ntVideoMode.VisScreenHeight, 
				gxVideoModes[i].ntVideoMode.Frequency));

			DISPDBG((10, "Linear %d, Compression %d",
				(gxVideoModes[i].ntVideoMode.DriverSpecificAttributeFlags & DRIVER_FLAG_LINEAR_MODE) ? 1 : 0,
				(gxVideoModes[i].ntVideoMode.DriverSpecificAttributeFlags & DRIVER_FLAG_COMPRESSION) ? 1 : 0));

		}
	}

	/* SAVE VALID MODE COUNT */

	gxVideoModeCount = curModeIndex;	

	return(TRUE);
}

/*-------------------------------------------------------------------------
 * GXStartIO
 *
 * This routine allows the VDD and NT to communicate with the miniport
 * driver.	It fields a variety of IOCTL requests and performs the requested
 * action or returns the desired information.
 -------------------------------------------------------------------------*/

BOOLEAN GXStartIO (PHW_DEVICE_EXTENSION HwDeviceExtension, PVIDEO_REQUEST_PACKET RequestPacket)
{
	VP_STATUS status;
	ULONG inIoSpace;
	PVIDEO_CLUT clutBuffer;
	PVOID virtualAddr;
	ULONG ulValue;

	DISPDBG((0, "StartIO, ioControl 0x%x,  DevExt 0x%x - %s\n",
		RequestPacket->IoControlCode, HwDeviceExtension, FindIoctl (RequestPacket->IoControlCode)));

	/* SWITCH ON THE I/O CONTROL CODE */
	
	switch (RequestPacket->IoControlCode) 
	{
		/* DPMS CONTROLS */

#if _WIN32_WINNT >= 0x0500
#if WIN2K_DPMS
			
        case IOCTL_VIDEO_GET_POWER_MANAGEMENT:   
		{
		    VIDEO_POWER_MANAGEMENT *powerState;	
			powerState = RequestPacket->OutputBuffer;
			if (RequestPacket->OutputBufferLength < sizeof(VIDEO_POWER_MANAGEMENT)) 
			{
			    RequestPacket->StatusBlock->Information = 0;
			    status = ERROR_INSUFFICIENT_BUFFER;
			    break;
			}
			powerState->PowerState  = HwDeviceExtension->CurrPowerState;
			powerState->Length      = HwDeviceExtension->CurrLength; 
			powerState->DPMSVersion = HwDeviceExtension->CurrDPMSVersion;
			
			/* NUMBER OF RETURNED BYTES */

			RequestPacket->StatusBlock->Information = sizeof(VIDEO_POWER_MANAGEMENT);
			status = NO_ERROR;
			break;
        }
			
		case IOCTL_VIDEO_SET_POWER_MANAGEMENT:  
		{
			VIDEO_POWER_MANAGEMENT *powerState;
			powerState = RequestPacket->InputBuffer;

			gfx_set_crt_enable ((int)powerState->PowerState);

			HwDeviceExtension->CurrPowerState  = powerState->PowerState;
			HwDeviceExtension->CurrLength      = powerState->Length;
			HwDeviceExtension->CurrDPMSVersion = powerState->DPMSVersion;
			break;
		}
#endif /* WIN2K_DPMS  */
#endif /* _WIN32_WINNT >= 0x0500 */
		
		case IOCTL_VIDEO_QUERY_PUBLIC_ACCESS_RANGES:
		{
			ULONG GXRegLength;
			VIDEO_PUBLIC_ACCESS_RANGES *publicAccessRange;

			DISPDBG((1, "GXStartIO - IOCTL_VIDEO_QUERY_PUBLIC_ACCESS_RANGES\n"));

			if (RequestPacket->OutputBufferLength <
					(RequestPacket->StatusBlock->Information =
					5*sizeof(VIDEO_PUBLIC_ACCESS_RANGES))) 
			{
				status = ERROR_INSUFFICIENT_BUFFER;
				break;
			}

			publicAccessRange = (VIDEO_PUBLIC_ACCESS_RANGES *)
					RequestPacket->OutputBuffer;

			/* REMAP ALL ADDRESSES */
			/* Even though we have already generated logical addresses for all */
			/* device memory, we must create a mapping that is compatible with */
			/* the display driver and not just the kernel.                     */

			/* MAP REGISTER POINTER */

			status = AddMappedAddressToArray (HwDeviceExtension->PhysGXRegAddr, 0x1000, 
				0, (PVOID)publicAccessRange, HwDeviceExtension);

			publicAccessRange++;

			/* MAP VIDEO POINTER */
	
			if (status == NO_ERROR)
			{
				status = AddMappedAddressToArray (HwDeviceExtension->PhysGXVidAddr, 0x1000, 
					0, (PVOID)publicAccessRange, HwDeviceExtension);
			}
			publicAccessRange++;

			/* MAP GP POINTER */

			if (status == NO_ERROR)
			{
				status = AddMappedAddressToArray (HwDeviceExtension->PhysGXGPAddr, 0x1000, 
					0, (PVOID)publicAccessRange, HwDeviceExtension);
			}
			publicAccessRange++;

			/* MAP VGA POINTER */

			if (status == NO_ERROR)
			{
				status = AddMappedAddressToArray (HwDeviceExtension->PhysGXVGAAddr, 31, 
					1, (PVOID)publicAccessRange, HwDeviceExtension);
			}
			publicAccessRange++;
			
			/* SET CPU SPEED */

			publicAccessRange->VirtualAddress = (void *)gfx_get_core_freq();			
			
			break;
		}

		case IOCTL_VIDEO_MAP_VIDEO_MEMORY:
		{
			VIDEO_MEMORY_INFORMATION *memoryInfo;
			ULONG frameLength;

			DISPDBG((1, "GXStartIO - MapVideoMemory\n"));

			if ((RequestPacket->OutputBufferLength <
					(RequestPacket->StatusBlock->Information =
					sizeof(VIDEO_MEMORY_INFORMATION))) ||
					(RequestPacket->InputBufferLength < sizeof(VIDEO_MEMORY))) 
			{
				DISPDBG((1, "output buffer too small\n"));
				status = ERROR_INSUFFICIENT_BUFFER;
				break;
			}
			memoryInfo = RequestPacket->OutputBuffer;

			DISPDBG((1, "Output ptr 0x%x\n", memoryInfo));			

			inIoSpace   = 0;
			virtualAddr = NULL;
			frameLength = HwDeviceExtension->FBSize;

			status = VideoPortMapMemory(HwDeviceExtension,
						HwDeviceExtension->PhysGXFBAddr,
						&frameLength,
						&inIoSpace,
						&virtualAddr);

			memoryInfo->VideoRamBase   = virtualAddr;
			memoryInfo->VideoRamLength = HwDeviceExtension->FBSize;
			DISPDBG((1, "status 0x%x, virtual address 0x%x\n",
				status, virtualAddr));

			/* FRAME BUFFER AND VIDEO RAM ARE EQUIVALENT */

			memoryInfo->FrameBufferBase   = memoryInfo->VideoRamBase;
			memoryInfo->FrameBufferLength = memoryInfo->VideoRamLength;

			DISPDBG((0, "GXStartIO - MapVideoMemory returns.\n"));
			break;
		}

		case IOCTL_VIDEO_UNMAP_VIDEO_MEMORY:

			DISPDBG((0, "GXStartIO - UnMapVideoMemory\n"));

			if (RequestPacket->InputBufferLength < sizeof(VIDEO_MEMORY)) 
			{
				status = ERROR_INSUFFICIENT_BUFFER;
				break;
			}
			status = VideoPortUnmapMemory(HwDeviceExtension,
										  ((PVIDEO_MEMORY)
										   (RequestPacket->InputBuffer))->
										  RequestedVirtualAddress,
										  0);

			break;

		case IOCTL_VIDEO_QUERY_CURRENT_MODE:
		{
			PGX_VIDEO_MODE_INFORMATION modeInfo;

			DISPDBG((0, "GXStartIO - QueryCurrentMode, index %d\n", gxModeIndex));

			modeInfo = RequestPacket->OutputBuffer;

			if (RequestPacket->OutputBufferLength <
					 sizeof(GX_VIDEO_MODE_INFORMATION)) 
			{
				RequestPacket->StatusBlock->Information = 0;
				status = ERROR_INSUFFICIENT_BUFFER;
			} 
			else 
			{
				/* STRUCTURE ASSIGNMENT (COPY) */

				RequestPacket->StatusBlock->Information =
					 sizeof(GX_VIDEO_MODE_INFORMATION);

				modeInfo->modeInfo       = gxVideoModes[gxModeIndex].ntVideoMode;
				modeInfo->OffscreenStart = GetOffscreenStart();
				modeInfo->OffscreenEnd   = GetOffscreenEnd();

				status = NO_ERROR;
			}

			break;
		}

		case IOCTL_VIDEO_QUERY_AVAIL_MODES:
		{
			VIDEO_MODE_INFORMATION *modeInfo;
			ULONG i;

			DISPDBG((1, "GXStartIO - QueryAvailableModes\n"));

			RequestPacket->StatusBlock->Information =
					gxVideoModeCount * sizeof(VIDEO_MODE_INFORMATION);

			DISPDBG((1, "modePtr %x, modes %d, buf size %d, need %d\n",
				RequestPacket->OutputBuffer,
				gxVideoModeCount,
				RequestPacket->OutputBufferLength,
				gxVideoModeCount * sizeof(VIDEO_MODE_INFORMATION)));

			if (RequestPacket->OutputBufferLength <
						(gxVideoModeCount * sizeof(VIDEO_MODE_INFORMATION))) 
			{
				status = ERROR_INSUFFICIENT_BUFFER;
			} 
			else 
			{
				/* COPY DATA */

				modeInfo = RequestPacket->OutputBuffer;
				for (i = 0; i < gxVideoModeCount; i++) 
				{
					modeInfo[i] = gxVideoModes[i].ntVideoMode;
				}

				/* DEBUG CHECK */

				DISPDBG((0, "gxVideoModes\n"));
				for(i = 0; i < gxVideoModeCount; i++) 
				{
					DISPDBG((0, "index %d, width %d, stride %d, length %d, rate %d\n",
						gxVideoModes[i].ntVideoMode.ModeIndex,
						gxVideoModes[i].ntVideoMode.VisScreenWidth,
						gxVideoModes[i].ntVideoMode.ScreenStride,
						gxVideoModes[i].ntVideoMode.Length,
						gxVideoModes[i].ntVideoMode.Frequency));
				}
				
				DISPDBG((0, "returned modes \n"));
				for(i = 0; i < gxVideoModeCount; i++) 
				{
					DISPDBG((0, "index %d, width %d, stride %d, length %d, rate %d\n",
						modeInfo[i].ModeIndex, modeInfo[i].VisScreenWidth,
						modeInfo[i].ScreenStride, modeInfo[i].Length, modeInfo[i].Frequency));
				}

				status = NO_ERROR;
			}
			break;
		}

		case IOCTL_VIDEO_QUERY_NUM_AVAIL_MODES:
		{
			VIDEO_NUM_MODES *videoNumModePtr;

			DISPDBG((0, "GXStartIO - QueryNumAvailableModes\n"));
			DISPDBG((0, "mode count %d\n", gxVideoModeCount));

			if (RequestPacket->OutputBufferLength < sizeof(VIDEO_NUM_MODES)) 
			{
				RequestPacket->StatusBlock->Information = 0;
				status = ERROR_INSUFFICIENT_BUFFER;
			} 
			else 
			{
				RequestPacket->StatusBlock->Information = sizeof(VIDEO_NUM_MODES);
				videoNumModePtr = (VIDEO_NUM_MODES *) RequestPacket->OutputBuffer;

				videoNumModePtr->NumModes = gxVideoModeCount;
				videoNumModePtr->ModeInformationLength = sizeof(VIDEO_MODE_INFORMATION) ;
				status = NO_ERROR;
			}

			break;
		}

		case IOCTL_VIDEO_SET_CURRENT_MODE:
		{
			ULONG modeIndex;

			DISPDBG((1, "GXStartIO - SetCurrentMode\n"));

			/* GRAB MODE INDEX */

			modeIndex = ((PVIDEO_MODE) (RequestPacket->InputBuffer))->RequestedMode;
			DISPDBG((1, "mode index %d\n", modeIndex));

			if (modeIndex < gxVideoModeCount) 
			{
				/* STORE MODE PARAMETERS */

				gxModeIndex = modeIndex;
				HwDeviceExtension->BitsPerPixel = gxVideoModes[gxModeIndex].ntVideoMode.BitsPerPlane;
				HwDeviceExtension->ScreenWidth  = gxVideoModes[gxModeIndex].ntVideoMode.VisScreenWidth;
				HwDeviceExtension->ScreenHeight = gxVideoModes[gxModeIndex].ntVideoMode.VisScreenHeight;
				HwDeviceExtension->VRefreshRate = gxVideoModes[gxModeIndex].ntVideoMode.Frequency;
				HwDeviceExtension->ScreenStride = gxVideoModes[gxModeIndex].ntVideoMode.ScreenStride;
				HwDeviceExtension->ModeFlags    = gxVideoModes[gxModeIndex].ntVideoMode.DriverSpecificAttributeFlags;

		
				DISPDBG((0, "BitsPerPixel %d\n", HwDeviceExtension->BitsPerPixel));
				DISPDBG((0, "ScreenWidth  %d\n", HwDeviceExtension->ScreenWidth));
				DISPDBG((0, "ScreenHeight %d\n", HwDeviceExtension->ScreenHeight));
				DISPDBG((0, "VRefreshRate %d\n", HwDeviceExtension->VRefreshRate));

				/* SET THE MODE ON THE HARDWARE */

				GXSetDisplayMode(HwDeviceExtension, (int)gxModeIndex);				

				status = NO_ERROR;
			} 
			else 
			{
				status = ERROR_INVALID_PARAMETER;
			}

			break;
		}

		case IOCTL_VIDEO_SET_COLOR_REGISTERS:

			DISPDBG((0, "\n\nGXStartIO - SetColorRegs\n"));

			clutBuffer = RequestPacket->InputBuffer;

			DISPDBG((0, "InputBufferLength %d\n", RequestPacket->InputBufferLength));
			DISPDBG((0, "NumEntries %d\n", clutBuffer->NumEntries));
			
			/* CHECK FOR ROOM IN THE INPUT BUFFER */

			if ((RequestPacket->InputBufferLength < sizeof(VIDEO_CLUT) -
				 sizeof(ULONG)) ||
				(RequestPacket->InputBufferLength < sizeof(VIDEO_CLUT) +
				 (sizeof(ULONG) * (clutBuffer->NumEntries - 1)))) 
			{
				status = ERROR_INSUFFICIENT_BUFFER;
				DISPDBG((300, "Set color registers, bad buffer\n"));
				break;
			}

			if (HwDeviceExtension->BitsPerPixel == 8) 
			{
				GXSetPalette(HwDeviceExtension,
						(PULONG) clutBuffer->LookupTable,
						clutBuffer->FirstEntry,
						clutBuffer->NumEntries);

				status = NO_ERROR;
			}
			else 
			{
				status = ERROR_INVALID_PARAMETER;
			}
			break;

		case IOCTL_VIDEO_QUERY_POINTER_CAPABILITIES:
		{
			PVIDEO_POINTER_CAPABILITIES pointerCaps = RequestPacket->OutputBuffer;

			DISPDBG((0, "GXStartIO - QueryPointerCapabilities\n"));
			if (RequestPacket->OutputBufferLength < sizeof(VIDEO_POINTER_CAPABILITIES))
			{
				RequestPacket->StatusBlock->Information = 0;
				status = ERROR_INSUFFICIENT_BUFFER;
				break;
			}

			pointerCaps->Flags = VIDEO_MODE_MONO_POINTER | VIDEO_MODE_ASYNC_POINTER;
			
			pointerCaps->MaxWidth  = CURSOR_WIDTH;
			pointerCaps->MaxHeight = CURSOR_HEIGHT;
			pointerCaps->HWPtrBitmapStart = 0;
			pointerCaps->HWPtrBitmapEnd   = 0;

			/* NUMBER OF BYTES RETURNED */

			RequestPacket->StatusBlock->Information = sizeof(VIDEO_POINTER_CAPABILITIES);

			status = NO_ERROR;
			break;
		}

		case IOCTL_VIDEO_RESET_DEVICE:

			DISPDBG((0, "GXStartIO - RESET_DEVICE\n"));

			if(HwDeviceExtension->Compression)
				gfx_set_compression_enable(0); 

			GXSetMode3 (HwDeviceExtension);

			status = NO_ERROR;
			break;

		case IOCTL_VIDEO_QUERY_COLOR_CAPABILITIES:
			
			/* UNSUPPORTED */
			status = ERROR_INVALID_FUNCTION;
			break;

		case IOCTL_VIDEO_SHARE_VIDEO_MEMORY:
		{
			PVIDEO_SHARE_MEMORY pShareMemory;
			PVIDEO_SHARE_MEMORY_INFORMATION pShareMemoryInformation;
			PHYSICAL_ADDRESS shareAddress;
			PVOID virtualAddress;
			ULONG sharedViewSize;
			ULONG inIoSpace;

			DISPDBG((1, "GXStartIO - IOCTL_VIDEO_SHARE_VIDEO_MEMORY\n"));

			if ( (RequestPacket->OutputBufferLength < sizeof(VIDEO_SHARE_MEMORY_INFORMATION)) ||
				 (RequestPacket->InputBufferLength < sizeof(VIDEO_MEMORY)) ) 
			{

				DISPDBG((1, "IOCTL_VIDEO_SHARE_VIDEO_MEMORY - insufficient buffer\n"));
				status = ERROR_INSUFFICIENT_BUFFER;
				break;
			}

			pShareMemory = RequestPacket->InputBuffer;

			if ((pShareMemory->ViewOffset > HwDeviceExtension->FBSize) ||
			   ((pShareMemory->ViewOffset + pShareMemory->ViewSize) >
					  HwDeviceExtension->FBSize) ) 
			{
				DISPDBG((0, "IOCTL_VIDEO_SHARE_VIDEO_MEMORY - invalid parameter\n"));
				status = ERROR_INVALID_PARAMETER;
				break;
			}

			RequestPacket->StatusBlock->Information = sizeof(VIDEO_SHARE_MEMORY_INFORMATION);

			/* BEWARE */
			/* The input buffer and the output buffer are the same buffer, and */
			/* therefore data should not be copied from one to the other.      */

			virtualAddress = pShareMemory->ProcessHandle;
			sharedViewSize = HwDeviceExtension->FBSize;
			inIoSpace      = 0;	

			shareAddress.QuadPart = HwDeviceExtension->PhysGXFBAddr.QuadPart;

			DISPDBG((1, "share addr (%x,%x), io %d, virtual %x, size %d\n",
					shareAddress.HighPart, shareAddress.LowPart,
					inIoSpace, virtualAddress, sharedViewSize));

			status = VideoPortMapMemory(HwDeviceExtension, shareAddress, &sharedViewSize, 
										&inIoSpace, &virtualAddress);

			DISPDBG((1, "status %x, sharedviewsize %x, virtual %x, viewOffset %x\n",
					status, sharedViewSize, virtualAddress, pShareMemory->ViewOffset));

			pShareMemoryInformation = RequestPacket->OutputBuffer;
    
			pShareMemoryInformation->SharedViewOffset = pShareMemory->ViewOffset;
			pShareMemoryInformation->VirtualAddress   = virtualAddress;
			pShareMemoryInformation->SharedViewSize   = sharedViewSize;
			break;
		}

		case IOCTL_VIDEO_UNSHARE_VIDEO_MEMORY:
		{
			PVIDEO_SHARE_MEMORY pShareMemory;
			DISPDBG((0, "GXStartIO - IOCTL_VIDEO_UNSHARE_VIDEO_MEMORY\n"));

			if (RequestPacket->InputBufferLength < sizeof(VIDEO_SHARE_MEMORY)) 
			{
				status = ERROR_INSUFFICIENT_BUFFER;
				break;
			}

			pShareMemory = RequestPacket->InputBuffer;

			/* UNMAP GX REGISTER SPACE */

			status = VideoPortUnmapMemory(HwDeviceExtension,
										  pShareMemory->RequestedVirtualAddress,
										  pShareMemory->ProcessHandle);

			DISPDBG((1, "Unshared addr %x, process %x\n",
					pShareMemory->RequestedVirtualAddress,
					pShareMemory->ProcessHandle));

			break;
		}

		case IOCTL_GET_DURANGO_FUNCS:
		{
			PDURANGO_FUNCS pDurFuncs;
			DISPDBG((1, "GXStartIO - IOCTL_GET_DURANGO_FUNCS\n"));

			status = NO_ERROR;

			if (RequestPacket->OutputBufferLength < sizeof(*pDurFuncs)) 
			{
				status = ERROR_INSUFFICIENT_BUFFER;
			}
			else
			{
				pDurFuncs = RequestPacket->OutputBuffer;
 
				/* ROUTINES IN GFX_DISP.C */

				pDurFuncs->pfn_is_display_mode_supported = gfx_is_display_mode_supported;
				pDurFuncs->pfn_set_display_mode = gfx_set_display_mode;
				pDurFuncs->pfn_set_display_timings = gfx_set_display_timings;
				pDurFuncs->pfn_set_display_pitch = gfx_set_display_pitch;
				pDurFuncs->pfn_set_display_offset = gfx_set_display_offset;
				pDurFuncs->pfn_set_display_palette_entry = gfx_set_display_palette_entry;
                pDurFuncs->pfn_set_display_palette_entry = gfx_set_display_palette_entry;
				pDurFuncs->pfn_set_display_palette = gfx_set_display_palette;
				pDurFuncs->pfn_set_clock_frequency = gfx_set_clock_frequency;
				pDurFuncs->pfn_set_crt_enable = gfx_set_crt_enable;
				pDurFuncs->pfn_set_cursor_enable = gfx_set_cursor_enable;
				pDurFuncs->pfn_set_cursor_colors = gfx_set_cursor_colors;
				pDurFuncs->pfn_set_cursor_position = gfx_set_cursor_position;
				pDurFuncs->pfn_set_cursor_shape32 = gfx_set_cursor_shape32;
                pDurFuncs->pfn_set_cursor_shape64 = gfx_set_cursor_shape64;
				pDurFuncs->pfn_set_compression_enable = gfx_set_compression_enable;
				pDurFuncs->pfn_set_compression_offset = gfx_set_compression_offset;
				pDurFuncs->pfn_set_compression_pitch = gfx_set_compression_pitch;
				pDurFuncs->pfn_set_compression_size = gfx_set_compression_size;
				pDurFuncs->pfn_test_timing_active = gfx_test_timing_active;
				pDurFuncs->pfn_test_vertical_active = gfx_test_vertical_active;
				pDurFuncs->pfn_wait_vertical_blank = gfx_wait_vertical_blank;
				pDurFuncs->pfn_delay_milliseconds = gfx_delay_milliseconds;
				pDurFuncs->pfn_delay_microseconds = gfx_delay_microseconds;
				pDurFuncs->pfn_enable_panning = gfx_enable_panning;
				pDurFuncs->pfn_set_fixed_timings = gfx_set_fixed_timings;

				/* "READ" ROUTINES IN GFX_DISP.C */

				pDurFuncs->pfn_get_display_details = gfx_get_display_details;
				pDurFuncs->pfn_get_display_pitch = gfx_get_display_pitch;
				pDurFuncs->pfn_get_sync_polarities = gfx_get_sync_polarities;
				pDurFuncs->pfn_get_clock_frequency = gfx_get_clock_frequency;
				pDurFuncs->pfn_get_max_supported_pixel_clock  = gfx_get_max_supported_pixel_clock ;
				pDurFuncs->pfn_mode_frequency_supported = gfx_mode_frequency_supported;
				pDurFuncs->pfn_get_refreshrate_from_frequency = gfx_get_refreshrate_from_frequency;
				pDurFuncs->pfn_get_display_mode_count = gfx_get_display_mode_count;
				pDurFuncs->pfn_get_display_mode = gfx_get_display_mode;
				pDurFuncs->pfn_get_hactive = gfx_get_hactive;
				pDurFuncs->pfn_get_hblank_start = gfx_get_hblank_start;
				pDurFuncs->pfn_get_hsync_start = gfx_get_hsync_start;
				pDurFuncs->pfn_get_hsync_end = gfx_get_hsync_end;
				pDurFuncs->pfn_get_hblank_end = gfx_get_hblank_end;
				pDurFuncs->pfn_get_htotal = gfx_get_htotal;
				pDurFuncs->pfn_get_vactive = gfx_get_vactive;
				pDurFuncs->pfn_get_vline = gfx_get_vline;
				pDurFuncs->pfn_get_vblank_start = gfx_get_vblank_start;
				pDurFuncs->pfn_get_vsync_start = gfx_get_vsync_start;
				pDurFuncs->pfn_get_vsync_end = gfx_get_vsync_end;
				pDurFuncs->pfn_get_vblank_end = gfx_get_vblank_end;
				pDurFuncs->pfn_get_vtotal = gfx_get_vtotal;
				pDurFuncs->pfn_get_display_bpp = gfx_get_display_bpp;
				pDurFuncs->pfn_get_display_offset = gfx_get_display_offset;
				pDurFuncs->pfn_get_display_palette_entry = gfx_get_display_palette_entry;
				pDurFuncs->pfn_get_display_palette = gfx_get_display_palette;
				pDurFuncs->pfn_get_cursor_enable = gfx_get_cursor_enable;
				pDurFuncs->pfn_get_cursor_offset = gfx_get_cursor_offset;
				pDurFuncs->pfn_get_cursor_position = gfx_get_cursor_position;
				pDurFuncs->pfn_get_cursor_clip = gfx_get_cursor_clip;
				pDurFuncs->pfn_get_cursor_color = gfx_get_cursor_color;
				pDurFuncs->pfn_get_compression_enable = gfx_get_compression_enable;
				pDurFuncs->pfn_get_compression_offset = gfx_get_compression_offset;
				pDurFuncs->pfn_get_compression_pitch = gfx_get_compression_pitch;
				pDurFuncs->pfn_get_compression_size = gfx_get_compression_size;
				pDurFuncs->pfn_get_valid_bit = gfx_get_valid_bit;

				/* ROUTINES IN GFX_RNDR.C */

				pDurFuncs->pfn_set_bpp = gfx_set_bpp;
				pDurFuncs->pfn_set_solid_pattern = gfx_set_solid_pattern;
				pDurFuncs->pfn_set_mono_pattern = gfx_set_mono_pattern;
				pDurFuncs->pfn_set_color_pattern = gfx_set_color_pattern;
				pDurFuncs->pfn_set_solid_source = gfx_set_solid_source;
				pDurFuncs->pfn_set_mono_source = gfx_set_mono_source;
				pDurFuncs->pfn_set_raster_operation = gfx_set_raster_operation;
				pDurFuncs->pfn_pattern_fill = gfx_pattern_fill;
				pDurFuncs->pfn_screen_to_screen_blt = gfx_screen_to_screen_blt;
				pDurFuncs->pfn_screen_to_screen_xblt = gfx_screen_to_screen_xblt;
				pDurFuncs->pfn_color_bitmap_to_screen_blt = gfx_color_bitmap_to_screen_blt;
				pDurFuncs->pfn_color_bitmap_to_screen_xblt = gfx_color_bitmap_to_screen_xblt;
				pDurFuncs->pfn_mono_bitmap_to_screen_blt = gfx_mono_bitmap_to_screen_blt;
				pDurFuncs->pfn_text_blt = gfx_text_blt;
				pDurFuncs->pfn_bresenham_line = gfx_bresenham_line;
				pDurFuncs->pfn_wait_until_idle = gfx_wait_until_idle;
				pDurFuncs->pfn_test_blt_pending = gfx_test_blt_pending;

				pDurFuncs->pfn2_set_source_stride = gfx2_set_source_stride;
				pDurFuncs->pfn2_set_destination_stride = gfx2_set_destination_stride ;
				pDurFuncs->pfn2_set_pattern_origin = gfx2_set_pattern_origin;
				pDurFuncs->pfn2_set_source_transparency = gfx2_set_source_transparency;
				pDurFuncs->pfn2_set_alpha_mode = gfx2_set_alpha_mode;
				pDurFuncs->pfn2_set_alpha_value = gfx2_set_alpha_value;
				pDurFuncs->pfn2_pattern_fill = gfx2_pattern_fill;
				pDurFuncs->pfn2_color_pattern_fill = gfx2_color_pattern_fill;
				pDurFuncs->pfn2_screen_to_screen_blt = gfx2_screen_to_screen_blt;
				pDurFuncs->pfn2_mono_expand_blt = gfx2_mono_expand_blt;
				pDurFuncs->pfn2_color_bitmap_to_screen_blt = gfx2_color_bitmap_to_screen_blt;
				pDurFuncs->pfn2_mono_bitmap_to_screen_blt = gfx2_mono_bitmap_to_screen_blt;
				pDurFuncs->pfn2_text_blt = gfx2_text_blt;
				pDurFuncs->pfn2_bresenham_line = gfx2_bresenham_line;

				/* ROUTINES IN GFX_VID.C */

				pDurFuncs->pfn_set_video_enable = gfx_set_video_enable;
				pDurFuncs->pfn_set_video_format = gfx_set_video_format;
				pDurFuncs->pfn_set_video_size = gfx_set_video_size;
				pDurFuncs->pfn_set_video_offset = gfx_set_video_offset;
				pDurFuncs->pfn_set_video_yuv_pitch = gfx_set_video_yuv_pitch;
				pDurFuncs->pfn_set_video_yuv_offsets = gfx_set_video_yuv_offsets;
				pDurFuncs->pfn_set_video_window = gfx_set_video_window;
				pDurFuncs->pfn_set_video_left_crop = gfx_set_video_left_crop;
				pDurFuncs->pfn_set_video_scale = gfx_set_video_scale;
				pDurFuncs->pfn_set_video_downscale_config = gfx_set_video_downscale_config;
				pDurFuncs->pfn_set_video_color_key = gfx_set_video_color_key;
				pDurFuncs->pfn_set_video_filter = gfx_set_video_filter;
				pDurFuncs->pfn_set_video_palette = gfx_set_video_palette;
                pDurFuncs->pfn_set_graphics_palette_entry = gfx_set_graphics_palette_entry;
				pDurFuncs->pfn_set_video_downscale_coefficients = gfx_set_video_downscale_coefficients;
				pDurFuncs->pfn_set_video_downscale_enable = gfx_set_video_downscale_enable;
				pDurFuncs->pfn_set_video_cursor = gfx_set_video_cursor;
				pDurFuncs->pfn_set_video_request = gfx_set_video_request;

				pDurFuncs->pfn_select_alpha_region = gfx_select_alpha_region;
				pDurFuncs->pfn_set_alpha_enable = gfx_set_alpha_enable;
				pDurFuncs->pfn_set_alpha_window = gfx_set_alpha_window;
				pDurFuncs->pfn_set_alpha_value = gfx_set_alpha_value;
				pDurFuncs->pfn_set_alpha_priority = gfx_set_alpha_priority;
				pDurFuncs->pfn_set_alpha_color = gfx_set_alpha_color;
				pDurFuncs->pfn_set_no_ck_outside_alpha = gfx_set_no_ck_outside_alpha;
				pDurFuncs->CursorOffset = gfx_get_frame_buffer_size() - 0x5000;
				
				pDurFuncs->pfn_detect_cpu = gfx_detect_cpu;
				pDurFuncs->pfn_get_cpu_register_base = gfx_get_cpu_register_base;
				pDurFuncs->pfn_get_frame_buffer_base = gfx_get_frame_buffer_base;
				pDurFuncs->pfn_get_frame_buffer_size = gfx_get_frame_buffer_size;
				pDurFuncs->pfn_get_vid_register_base = gfx_get_vid_register_base;

				pDurFuncs->pPanelWidth  = &PanelWidth;
				pDurFuncs->pPanelHeight = &PanelHeight;
				pDurFuncs->pPanelEnable = &PanelEnable;
				pDurFuncs->pDeltaX = &DeltaX;
				pDurFuncs->pDeltaY = &DeltaY;
				pDurFuncs->LinearFB = HwDeviceExtension->LinearFB;
				pDurFuncs->AntialiasedText = HwDeviceExtension->AntialiasedText;
										
				RequestPacket->StatusBlock->Information = sizeof(*pDurFuncs);
            }
		}
		break;

		case IOCTL_SET_HEAP_INFORMATION:
		{
            ULONG *heapRange;
			DISPDBG((1, "GXStartIO - IOCTL_SET_HEAP_CALLBACK\n"));

			status = NO_ERROR;

			if (RequestPacket->InputBufferLength < 8) 
			{
				status = ERROR_INSUFFICIENT_BUFFER;
			}
			else
			{
                heapRange = RequestPacket->OutputBuffer;
                heapStart = heapRange[0];
                heapEnd   = heapRange[1];

                RequestPacket->StatusBlock->Information = 0;
			    
                DISPDBG ((1000, "Heap Range 0x%8.8lX - 0x%8.8lX\n", heapStart, heapEnd));
            }
		}
		break;

		case IOCTL_GET_REGISTRY_PARAM:
		{
			RegistryData *regData;
			DISPDBG((1, "GXStartIO - IOCTL_GET_REGISTRY_PARAM\n"));

			if (RequestPacket->OutputBufferLength < sizeof(RegistryData)) 
			{
				status = ERROR_INSUFFICIENT_BUFFER;
			}
			else
			{
				/* COMPRESSION */

				regData = RequestPacket->OutputBuffer;
				if(regData->Flag & REG_COMPRESSION)
				{
					if (VideoPortGetRegistryParameters(HwDeviceExtension,
													   L"Compression",
													   FALSE,
													   GeodeRegistryCallback,
													   &ulValue) == NO_ERROR) 
					{
						regData->Compression = ulValue;
					} 
					else 
					{
						regData->Compression = 0;
					}
					DISPDBG((2000, "Compression - %d\n", regData->Compression));
				} 

				/* DDC */
				
				if(regData->Flag & REG_DDCENABLED)
				{
					if (VideoPortGetRegistryParameters(HwDeviceExtension,
													   L"DDCEnabled",
													   FALSE,
													   GeodeRegistryCallback,
													   &ulValue) == NO_ERROR) 
					{
						regData->DDCEnabled = ulValue;
					} 
					else 
					{
						regData->DDCEnabled = 0;
					}
					DISPDBG((2000, "DDCReg - %d\n", regData->DDCEnabled));
				}
				if(regData->Flag & REG_FBLINEAR)
				{
					if (VideoPortGetRegistryParameters(HwDeviceExtension,
													   L"FBLinear",
													   FALSE,
													   GeodeRegistryCallback,
													   &ulValue) == NO_ERROR) 
					{
						regData->FBLinear = ulValue;
					} 
					else 
					{
						regData->FBLinear = 0;
					}
					DISPDBG((2000, "Linear FB - %d\n", regData->FBLinear));
				} 

				if(regData->Flag & REG_ANTIALIASED)
				{
					if (VideoPortGetRegistryParameters(HwDeviceExtension,
													   L"AntialiasedText",
													   FALSE,
													   GeodeRegistryCallback,
													   &ulValue) == NO_ERROR) 
					{
						regData->AntialiasedText = ulValue;
					} 
					else 
					{
						regData->AntialiasedText = 0;
					}
					DISPDBG((2000, "Antialiased Text - %d\n", regData->AntialiasedText));
				} 
				
				status = NO_ERROR;
				RequestPacket->StatusBlock->Information = sizeof(RegistryData);
			}
		}
		break;
		    
	    case IOCTL_SET_REGISTRY_PARAM:
		{
			RegistryData *regData;
			DISPDBG((1, "GXStartIO - IOCTL_SET_REGISTRY_PARAM\n"));

			if (RequestPacket->InputBufferLength < sizeof(RegistryData)) 
			{
			    status = ERROR_INSUFFICIENT_BUFFER;
			}
			else
			{
				regData = RequestPacket->InputBuffer;
				if (regData->Flag & REG_COMPRESSION)
				{
					VideoPortSetRegistryParameters (HwDeviceExtension,
												   L"Compression",
												   &regData->Compression,
												   sizeof(ULONG));

					DISPDBG((2000, "Compression - %d\n", regData->Compression));
				} 
				if(regData->Flag & REG_DDCENABLED)
				{
					VideoPortSetRegistryParameters(HwDeviceExtension,
												   L"DDCEnabled",
												   &regData->DDCEnabled,
												   sizeof(ULONG));

					DISPDBG((2000, "DDCReg - %d\n", regData->DDCEnabled));
				} 
				if (regData->Flag & REG_FBLINEAR)
				{
					VideoPortSetRegistryParameters (HwDeviceExtension,
												   L"FBLinear",
												   &regData->FBLinear,
												   sizeof(ULONG));

					DISPDBG((2000, "FBLinear - %d\n", regData->FBLinear));
				} 
				if (regData->Flag & REG_ANTIALIASED)
				{
					VideoPortSetRegistryParameters (HwDeviceExtension,
												   L"AntialiasedText",
												   &regData->AntialiasedText,
												   sizeof(ULONG));

					DISPDBG((2000, "Antialiased - %d\n", regData->AntialiasedText));
				} 
				status = NO_ERROR;
				RequestPacket->StatusBlock->Information = sizeof(RegistryData);
			}
		}
		
		break;

		/* INVALID IO CONTROL CODE */

		default:
			DISPDBG((0, "Fell through GX startIO routine - invalid command, %d\n",
				RequestPacket->IoControlCode));
			DISPDBG((0, "What? %d\n", IOCTL_VIDEO_QUERY_COLOR_CAPABILITIES));
			status = ERROR_INVALID_FUNCTION;
			break;

	}

	RequestPacket->StatusBlock->Status = status;

	DISPDBG((0, "GXStartIO returning...\n"));
	return (TRUE);

} /* GXStartIO */

BOOLEAN GXInitialize(PHW_DEVICE_EXTENSION HwDeviceExtension)
{
	DISPDBG((0, "GXInitialize: HwDeviceExtension 0x%x\n", HwDeviceExtension));
	return(TRUE);
}

/*----------------------------------------------------------------------------
 * GXSetPalette
 *
 * Initializes the palette in an 8BPP mode.
 ----------------------------------------------------------------------------*/

void GXSetPalette(
	HW_DEVICE_EXTENSION *HwDeviceExtension,
	ULONG *pPal,
	ULONG startIndex,
	ULONG count)
{
	ULONG i;
	ULONG value;
	ULONG *palIndex;
    ULONG data24;

	palIndex = pPal + startIndex;
	
	for(i = 0; i < count; i++) 
	{
		/* SWITCH BGR DATA TO RGB */

	    UCHAR R = (UCHAR)  (*palIndex & 0xFF);
	    UCHAR G = (UCHAR)  ((*palIndex & 0xFF00) >> 8);
	    UCHAR B = (UCHAR)  ((*palIndex & 0xFF0000) >> 16);
		
		data24 = (R << 16) | (G << 8) | B;

		gfx_set_display_palette_entry(startIndex + i, data24);
	    ++palIndex;
			
		DISPDBG((1, "set palette %x, %X\n", startIndex+i, data24));
	}

	DISPDBG((0, "GXSetPalette returns\n"));
}

#if _WIN32_WINNT >= 0x0500
#if WIN2K_DPMS

/*----------------------------------------------------------------------------
 * SetPanelEnable
 *----------------------------------------------------------------------------*/

void SetPanelEnable (HW_DEVICE_EXTENSION* HwDeviceExtension, int enable)
{
	unsigned long power;

	/* NO WORK IF NO PANEL PRESENT */

	if (HwDeviceExtension->PanelWidth == 0)
		return;

	power = READ_VID32 (RCDF_POWER_MANAGEMENT);

	if (enable) power |=  RCDF_PM_PANEL_POWER_ON;
	else        power &= ~RCDF_PM_PANEL_POWER_ON;

	WRITE_VID32 (RCDF_POWER_MANAGEMENT, power);

	/* WAIT FOR THE PANEL TO BE FULLY POWERED OFF */

	if (!enable)
	{
		while (!(READ_VID32 (RCDF_POWER_MANAGEMENT) & 2))
			;
	}
}

VOID GXI2CWriteClockLine (PVOID HwDeviceExtension, UCHAR data)
{
    HW_DEVICE_EXTENSION *pDev = (HW_DEVICE_EXTENSION *)HwDeviceExtension;
    unsigned short address;
    unsigned long value;

    value = data ? DDC_CLK_HIGH : DDC_CLK_LOW;
    
    /* WRITE CLOCK DATA */

    address = (unsigned short)pDev->DDCBase + DDC_CLOCK_OUT;

    _asm {
        mov    eax, value
        mov    dx, address
        out    dx, eax        
    }
}

VOID GXI2CWriteDataLine (PVOID HwDeviceExtension, UCHAR data)
{
    HW_DEVICE_EXTENSION *pDev = (HW_DEVICE_EXTENSION *)HwDeviceExtension;
    unsigned short address;
    unsigned long value;

    value = data ? DDC_DATA_HIGH : DDC_DATA_LOW;   
    
    /* WRITE DATA */

    address = (unsigned short)pDev->DDCBase + DDC_DATA_OUT;

    _asm {
        mov    eax, value
        mov    dx, address
        out    dx, eax        
    }
}

BOOLEAN GXI2CReadClockLine (PVOID HwDeviceExtension)
{
    HW_DEVICE_EXTENSION *pDev = (HW_DEVICE_EXTENSION *)HwDeviceExtension;
    unsigned short address;
    unsigned long value;    
    
    /* READ CLOCK */

    address = (unsigned short)pDev->DDCBase + DDC_CLOCK_IN;

    _asm {

        mov    dx, address
        in     eax, dx
        mov    value, eax
    }

    if (value & DDC_CLK_HIGH)
        return TRUE;

    return FALSE;
}

BOOLEAN GXI2CReadDataLine (PVOID HwDeviceExtension)
{
    HW_DEVICE_EXTENSION *pDev = (HW_DEVICE_EXTENSION *)HwDeviceExtension;
    unsigned short address;
    unsigned long value;    
    
    /* READ DATA */

    address = (unsigned short)pDev->DDCBase + DDC_DATA_IN;

    _asm {

        mov    dx, address
        in     eax, dx
        mov    value, eax
    }

    if (value & DDC_DATA_HIGH)
        return TRUE;

    return FALSE;
}

/*----------------------------------------------------------------------------
 * GXGetChildDescriptor
 *----------------------------------------------------------------------------*/

ULONG GXGetChildDescriptor (
	HW_DEVICE_EXTENSION    *HwDeviceExtension, 
	PVIDEO_CHILD_ENUM_INFO  ChildEnumInfo,
	PVIDEO_CHILD_TYPE       pChildType, 
	PVOID                   pChildDescriptor, 
	PULONG                  pUId, 
	PULONG                  pUnused )
{
    BOOLEAN bGotEdid;
    I2C_CALLBACKS i2cCallbacks;
    DDC_CONTROL ddcControl;
    unsigned long edid[256];
    unsigned long base;
    
    switch (ChildEnumInfo->ChildIndex) 
    {
		/* ACPI                                                            */
		/* Case 0 is used to enumerate devices found by the ACPI firmware. */
        /* We do not currently support ACPI devices                        */

        case 0:

            return ERROR_NO_MORE_DEVICES;

		/* TREAT CASE 1 AS MONITOR        */
		/* Force non-DDC monitor for now. */
        
		case 1:

			*pChildType = Monitor;
			*pUId = P2_NONDDC_MONITOR;

            if (HwDeviceExtension->DDCEnabled)
            {
                /* READ THE IO BASE FOR GPIOS */

                _asm
                {
                    mov     dx, 0xCF8
                    mov     eax, 0x80007814
                    out     dx, eax
                    mov     dx, 0xCFC
                    in      eax, dx
                    mov     base, eax
                }

                base &= 0xFFFFFFFC;
                DISPDBG((5000, "DDC Base = 0x%X\n", base));

                if (base != 0 && base != 0xFFFFFFFF)
                {
                    /* ENABLE CLOCK AND DATA INPUTS AND OUTPUTS */

                    _asm {
                        mov   edx, base
                        add   dx,  DDC_OUT_ENABLE
                        mov   eax, DDC_DATA_HIGH
                        or    eax, DDC_CLK_HIGH
                        out   dx,  eax
                        mov   edx, base
                        add   dx,  DDC_IN_ENABLE
                        out   dx,  eax
                    }

                    HwDeviceExtension->DDCBase = base;
                    HwDeviceExtension->DDCClockMode = DDC_MODE_UNDEFINED;
                    HwDeviceExtension->DDCDataMode  = DDC_MODE_UNDEFINED;
            
                    ddcControl.Size = sizeof(DDC_CONTROL);
                    ddcControl.I2CCallbacks.WriteClockLine = GXI2CWriteClockLine;
                    ddcControl.I2CCallbacks.WriteDataLine  = GXI2CWriteDataLine;
                    ddcControl.I2CCallbacks.ReadClockLine  = GXI2CReadClockLine;
                    ddcControl.I2CCallbacks.ReadDataLine   = GXI2CReadDataLine;
                    ddcControl.EdidSegment = 0;

                    bGotEdid = VideoPortDDCMonitorHelper (
                        HwDeviceExtension,
                        &ddcControl,
                        pChildDescriptor,
                        ChildEnumInfo->ChildDescriptorSize);

                    DISPDBG((5000, "DDC Helper returned %s\n", bGotEdid ? "TRUE" : "FALSE"));
                }
            }

			return ERROR_MORE_DATA;

        default:

            return ERROR_NO_MORE_DEVICES;
    }
}

/*----------------------------------------------------------------------------
 * GXGetPowerState
 * 
 * Returns power state information.
 *----------------------------------------------------------------------------*/

VP_STATUS GXGetPowerState (	
	HW_DEVICE_EXTENSION     *HwDeviceExtension, 
	ULONG                    HwId,
	PVIDEO_POWER_MANAGEMENT  VideoPowerControl)
{
    VP_STATUS status;
    PHW_DEVICE_EXTENSION hwDeviceExtension = HwDeviceExtension;
    

    DISPDBG((2000, "GXGetPowrState: hwId(%xh) state = %d\n", 
                     (int)HwId, (int)VideoPowerControl->PowerState));

    switch((int)HwId)
    {
        case P2_DDC_MONITOR:
        case P2_NONDDC_MONITOR:

            switch (VideoPowerControl->PowerState)
            {
				/* WE SUPPORT ALL POWER STATE */

                case VideoPowerOn:
                case VideoPowerStandBy:
                case VideoPowerSuspend:
                case VideoPowerOff:
                case VideoPowerHibernate:
                case VideoPowerShutdown:
					status = NO_ERROR;

                    break;

                default:

                    DISPDBG((2000, "GXGetPowerState: Unknown monitor PowerState(%xh)\n", 
                                    (int)VideoPowerControl->PowerState));

                    ASSERT(FALSE);
                    status = ERROR_INVALID_PARAMETER;
            }
            break;

        case DISPLAY_ADAPTER_HW_ID:

            /* WE SUPPORT ALL POWER STATES */

            switch (VideoPowerControl->PowerState)
            {
                case VideoPowerOn:
                case VideoPowerStandBy:
                case VideoPowerSuspend:
                case VideoPowerHibernate:
                case VideoPowerShutdown:
                case VideoPowerOff:
                    status = NO_ERROR;
                    break;

                default:

                    DISPDBG((2000, "GXGetPowerState: Unknown adapter PowerState(%xh)\n", 
                                 (int)VideoPowerControl->PowerState));

                    ASSERT(FALSE);

                    status = ERROR_INVALID_PARAMETER;
            }
			
            break;

        default:

            DISPDBG((2000, "GXGetPowerState: Unknown hwId(%xh)", 
                            (int)HwId));
            ASSERT(FALSE);

            status = ERROR_INVALID_PARAMETER;
    }

    DISPDBG((2000, "GXGetPowerState: returning %xh\n", status));

    return(status);
}

/*----------------------------------------------------------------------------
 * GXGetPowerState
 * 
 * Sets the power state for a given device.
 *----------------------------------------------------------------------------*/

VP_STATUS GXSetPowerState (	
	HW_DEVICE_EXTENSION     *HwDeviceExtension, 
	ULONG                    HwId,
	PVIDEO_POWER_MANAGEMENT  VideoPowerControl)
{
    PHW_DEVICE_EXTENSION hwDeviceExtension = HwDeviceExtension;
    VP_STATUS status;

    DISPDBG((2000, "GXSetPowerState: hwId(%xh) state = %d\n", 
                   (int)HwId, (int)VideoPowerControl->PowerState));

    switch((int)HwId)
    {

        case P2_DDC_MONITOR:
        case P2_NONDDC_MONITOR:

            
            switch (VideoPowerControl->PowerState)
            {

                case VideoPowerHibernate:
                case VideoPowerShutdown:

                    DISPDBG((2000, "GXSetPowerState: monitor HIBERNATE\n"));

                    /* MONITOR STAYS ON FOR HIBERNATE AND SHUTDOWN */

                    status = NO_ERROR;
                    break;

                case VideoPowerOn:

                    DISPDBG((2000, "GXSetPowerState: monitor ON\n"));

					/* ENABLE THE MONITOR (AND THE PANEL) */

					gfx_set_crt_enable (CRT_ENABLE);
					SetPanelEnable (HwDeviceExtension, 1);

                    status = NO_ERROR;
                    break;

                case VideoPowerStandBy:

                    DISPDBG((2000, "GXSetPowerState: monitor STANDBY\n"));

                    /* HSYNC OFF, VSYNC ON */
					/* As panels have very little latency, we also disable */
					/* the panel.                                          */
                    
					SetPanelEnable(HwDeviceExtension, 0);
					gfx_set_crt_enable (CRT_STANDBY);
					
                    status = NO_ERROR;
                    break;

                case VideoPowerSuspend:

                    DISPDBG((2000, "GXSetPowerState: monitor SUSPEND\n"));

                    /* HSYNC ON, VSYNC OFF */
					/* As panels have very little latency, we also disable */
					/* the panel.                                          */

					SetPanelEnable (HwDeviceExtension, 0);
					gfx_set_crt_enable (CRT_STANDBY);
					
                    status = NO_ERROR;
                    break;

                case VideoPowerOff:

                    DISPDBG((2000, "GXSetPowerState: monitor OFF\n"));

					/* DISABLE THE CRT (AND THE PANEL)*/

					SetPanelEnable (HwDeviceExtension, 0);
                    gfx_set_crt_enable (CRT_DISABLE);

                    status = NO_ERROR;
                    break;

                default:

                    DISPDBG((2000, "GXGetPowerState: Unknown monitor PowerState(%xh)\n", 
                                     (int)VideoPowerControl->PowerState));

                    ASSERT(FALSE);
                    status = ERROR_INVALID_PARAMETER;
            }

            /* ALL WORK IS ACCOMPLISHED BY THIS POINT */

            break;

        case DISPLAY_ADAPTER_HW_ID:

			/* POWER STATES FOR INTERNAL VIDEO */
			/* As we have no real control over the power state of the internal */
			/* graphics controller, we will do nothing for these states and    */
			/* depend on the BIOS to disable the hardware.  Also, when leaving */
			/* a save-to-RAM situation, Windows passes a DrvAssertMode call    */
			/* that we use to set up all the hardware configuration that was   */
			/* lost.  So, we have no need to save any registers.  (for now)    */

            switch (VideoPowerControl->PowerState)
            {
                case VideoPowerHibernate:
                case VideoPowerShutdown:
					
                    DISPDBG((2000, "GXSetPowerState: adapter HIBERNATE\n"));			

					// SAVE FRAME BUFFER DATA
					// Windows expects all memory to be preserved when hibernating 
					// including offscreen bitmaps.  The display driver is responsible
                    // for specifying the range of valid memory before the system 
                    // hibernates.
                    //
                    if ((long)(heapEnd - heapStart) > 0)
                    {
					    fb_ptr = VideoPortAllocatePool (HwDeviceExtension, VpPagedPool, 
						    heapEnd - heapStart, 0x20786744);

                        if (fb_ptr != NULL)
                        {
                            from_hibernate = 1;
                            
                            memcpy (fb_ptr, gfx_virt_fbptr + heapStart, heapEnd - heapStart);
                        }
                        else
                        {
                            DISPDBG ((3000, "Unable to allocate space for video memory\n"));
                        }
                    }

					status = NO_ERROR;
                    break;

                case VideoPowerOn:

                    DISPDBG((2000, "GXSetPowerState: adapter ON\n")); 

					// RESTORE FRAME BUFFER DATA IF WE SUCCESSFULLY SAVED THE ENTIRE THING 

					if (from_hibernate)
					{
						from_hibernate = 0;
						memcpy (gfx_virt_fbptr + heapStart, fb_ptr, heapEnd - heapStart);

						VideoPortFreePool (HwDeviceExtension, fb_ptr);
						fb_ptr = NULL;
					}
                    status = NO_ERROR;
                    break;

                case VideoPowerStandBy:

                    DISPDBG((2000, "GXSetPowerState: adapter STANDBY\n"));
                    status = NO_ERROR;
                    break;

                case VideoPowerSuspend:

                    DISPDBG((2000, "GXSetPowerState: adapter SUSPEND\n"));
					status = NO_ERROR;
                    break;
    
                case VideoPowerOff:

                    DISPDBG((2000, "GXSetPowerState: adapter OFF\n"));
					status = NO_ERROR;
                    break;

                default:

                    DISPDBG((2000, "GXGetPowerState: Unknown adapter PowerState(%xh)\n", 
                                     (int)VideoPowerControl->PowerState));

                    ASSERT(FALSE);
                    status = ERROR_INVALID_PARAMETER;
            }

            break;
    
        default:

            DISPDBG((2000, "GXSetPowerState: Unknown hwId(%xh)\n", 
                             (int)HwId));

            ASSERT(FALSE);
            status = ERROR_INVALID_PARAMETER;
    }

    if( status == NO_ERROR)
	{
		HwDeviceExtension->PreviousPowerState = VideoPowerControl->PowerState;
		HwDeviceExtension->CurrPowerState     = VideoPowerControl->PowerState;
		HwDeviceExtension->CurrLength         = VideoPowerControl->Length;
		HwDeviceExtension->CurrDPMSVersion    = VideoPowerControl->DPMSVersion;
	}

    DISPDBG((2000, "GXSetPowerState: returning %xh\n", status));

    return(status);

}

#endif // WIN2K_DPMS
#endif //_WIN32_WINNT >= 0x500

int GXMapPhysicalAddress (PVOID HwDeviceExtension, ULONG address, 
						  ULONG size, UCHAR IO, PULONG ptr_out)
{
	PHYSICAL_ADDRESS temp_address;

	/* CHECK FOR NULL ADDRESS */
	/* Address == 0 implies that the address should not be mapped */

	if (address == 0 || ptr_out == NULL || size == 0)
		return 0;

	DISPDBG((1, "GXFindAdapter: Mapping register address 0x%x, length 0x%x\n",
			address, size));

	temp_address.LowPart  = address;
	temp_address.HighPart = 0;

	*ptr_out = (ULONG) VideoPortGetDeviceBase(HwDeviceExtension, temp_address, size, IO);
			
	if (*ptr_out == 0)		
	{
		DISPDBG((1, "Failed to map address\n"));
		return 1;
	}

	return 0;
}

VP_STATUS AddMappedAddressToArray (PHYSICAL_ADDRESS address, ULONG length, ULONG IO, 
								   PVOID array, PVOID HwDeviceExtension) 
{
	PVOID virtualAddr = NULL;
	VP_STATUS status = NO_ERROR;
	VIDEO_PUBLIC_ACCESS_RANGES *accessRanges = (VIDEO_PUBLIC_ACCESS_RANGES *)array;

	if (address.LowPart != 0)
	{
		DISPDBG((1, "Phys Addr 0x%x,0x%x, length 0x%x\n",
				(address).HighPart,
				(address).LowPart,
				length));

		status = VideoPortMapMemory (HwDeviceExtension, address, &length, &IO, &virtualAddr);
		
		accessRanges->VirtualAddress  = (void *) virtualAddr;
		accessRanges->InIoSpace       = IO;
		accessRanges->MappedInIoSpace = IO;
		
		DISPDBG((1, "VideoPortMapMemory status 0x%x, addr 0x%x\n",
					status, virtualAddr));
	}

	return status;
}

#if DBG

typedef struct tagVideoString
{
    char  *string;
    ULONG  code;

} VIDEO_STRING;

VIDEO_STRING allStrings[] = 
{
    { "IOCTL_VIDEO_ENABLE_VDM",                         IOCTL_VIDEO_ENABLE_VDM },
    { "IOCTL_VIDEO_DISABLE_VDM",                        IOCTL_VIDEO_DISABLE_VDM },
    { "IOCTL_VIDEO_REGISTER_VDM",                       IOCTL_VIDEO_REGISTER_VDM },
    { "IOCTL_VIDEO_SET_OUTPUT_DEVICE_POWER_STATE",      IOCTL_VIDEO_SET_OUTPUT_DEVICE_POWER_STATE },
    { "IOCTL_VIDEO_GET_OUTPUT_DEVICE_POWER_STATE",      IOCTL_VIDEO_GET_OUTPUT_DEVICE_POWER_STATE },
    { "IOCTL_VIDEO_MONITOR_DEVICE",                     IOCTL_VIDEO_MONITOR_DEVICE },
    { "IOCTL_VIDEO_ENUM_MONITOR_PDO",                   IOCTL_VIDEO_ENUM_MONITOR_PDO },
    { "IOCTL_VIDEO_INIT_WIN32K_CALLBACKS",              IOCTL_VIDEO_INIT_WIN32K_CALLBACKS },
    { "IOCTL_VIDEO_HANDLE_VIDEOPARAMETERS",             IOCTL_VIDEO_HANDLE_VIDEOPARAMETERS },
    { "IOCTL_VIDEO_IS_VGA_DEVICE",                      IOCTL_VIDEO_IS_VGA_DEVICE },
    { "IOCTL_VIDEO_SAVE_HARDWARE_STATE",                IOCTL_VIDEO_SAVE_HARDWARE_STATE },
    { "IOCTL_VIDEO_RESTORE_HARDWARE_STATE",             IOCTL_VIDEO_RESTORE_HARDWARE_STATE },
    { "IOCTL_VIDEO_QUERY_AVAIL_MODES",                  IOCTL_VIDEO_QUERY_AVAIL_MODES     },
    { "IOCTL_VIDEO_QUERY_NUM_AVAIL_MODES",              IOCTL_VIDEO_QUERY_NUM_AVAIL_MODES     },
    { "IOCTL_VIDEO_QUERY_CURRENT_MODE",                 IOCTL_VIDEO_QUERY_CURRENT_MODE },
    { "IOCTL_VIDEO_SET_CURRENT_MODE",                   IOCTL_VIDEO_SET_CURRENT_MODE },
    { "IOCTL_VIDEO_RESET_DEVICE",                       IOCTL_VIDEO_RESET_DEVICE },
    { "IOCTL_VIDEO_LOAD_AND_SET_FONT",                  IOCTL_VIDEO_LOAD_AND_SET_FONT },
    { "IOCTL_VIDEO_SET_PALETTE_REGISTERS",              IOCTL_VIDEO_SET_PALETTE_REGISTERS },
    { "IOCTL_VIDEO_SET_COLOR_REGISTERS",                IOCTL_VIDEO_SET_COLOR_REGISTERS },
    { "IOCTL_VIDEO_ENABLE_CURSOR",                      IOCTL_VIDEO_ENABLE_CURSOR },
    { "IOCTL_VIDEO_DISABLE_CURSOR",                     IOCTL_VIDEO_DISABLE_CURSOR },
    { "IOCTL_VIDEO_SET_CURSOR_ATTR",                    IOCTL_VIDEO_SET_CURSOR_ATTR },
    { "IOCTL_VIDEO_QUERY_CURSOR_ATTR",                  IOCTL_VIDEO_QUERY_CURSOR_ATTR },
    { "IOCTL_VIDEO_SET_CURSOR_POSITION",                IOCTL_VIDEO_SET_CURSOR_POSITION },
    { "IOCTL_VIDEO_QUERY_CURSOR_POSITION",              IOCTL_VIDEO_QUERY_CURSOR_POSITION },
    { "IOCTL_VIDEO_ENABLE_POINTER",                     IOCTL_VIDEO_ENABLE_POINTER },
    { "IOCTL_VIDEO_DISABLE_POINTER",                    IOCTL_VIDEO_DISABLE_POINTER },
    { "IOCTL_VIDEO_SET_POINTER_ATTR",                   IOCTL_VIDEO_SET_POINTER_ATTR },
    { "IOCTL_VIDEO_QUERY_POINTER_ATTR",                 IOCTL_VIDEO_QUERY_POINTER_ATTR },
    { "IOCTL_VIDEO_SET_POINTER_POSITION",               IOCTL_VIDEO_SET_POINTER_POSITION },
    { "IOCTL_VIDEO_QUERY_POINTER_POSITION",             IOCTL_VIDEO_QUERY_POINTER_POSITION },
    { "IOCTL_VIDEO_QUERY_POINTER_CAPABILITIES",         IOCTL_VIDEO_QUERY_POINTER_CAPABILITIES },
    { "IOCTL_VIDEO_GET_BANK_SELECT_CODE",               IOCTL_VIDEO_GET_BANK_SELECT_CODE },
    { "IOCTL_VIDEO_MAP_VIDEO_MEMORY",                   IOCTL_VIDEO_MAP_VIDEO_MEMORY },
    { "IOCTL_VIDEO_UNMAP_VIDEO_MEMORY",                 IOCTL_VIDEO_UNMAP_VIDEO_MEMORY },
    { "IOCTL_VIDEO_QUERY_PUBLIC_ACCESS_RANGES",         IOCTL_VIDEO_QUERY_PUBLIC_ACCESS_RANGES },
    { "IOCTL_VIDEO_FREE_PUBLIC_ACCESS_RANGES",          IOCTL_VIDEO_FREE_PUBLIC_ACCESS_RANGES },
    { "IOCTL_VIDEO_QUERY_COLOR_CAPABILITIES",           IOCTL_VIDEO_QUERY_COLOR_CAPABILITIES },
    { "IOCTL_VIDEO_SET_POWER_MANAGEMENT",               IOCTL_VIDEO_SET_POWER_MANAGEMENT },
    { "IOCTL_VIDEO_GET_POWER_MANAGEMENT",               IOCTL_VIDEO_GET_POWER_MANAGEMENT },
    { "IOCTL_VIDEO_SHARE_VIDEO_MEMORY",                 IOCTL_VIDEO_SHARE_VIDEO_MEMORY },
    { "IOCTL_VIDEO_UNSHARE_VIDEO_MEMORY",               IOCTL_VIDEO_UNSHARE_VIDEO_MEMORY },
    { "IOCTL_VIDEO_SET_COLOR_LUT_DATA",                 IOCTL_VIDEO_SET_COLOR_LUT_DATA },
    { "IOCTL_VIDEO_GET_CHILD_STATE",                    IOCTL_VIDEO_GET_CHILD_STATE },
    { "IOCTL_VIDEO_VALIDATE_CHILD_STATE_CONFIGURATION", IOCTL_VIDEO_VALIDATE_CHILD_STATE_CONFIGURATION },
    { "IOCTL_VIDEO_SET_CHILD_STATE_CONFIGURATION",      IOCTL_VIDEO_SET_CHILD_STATE_CONFIGURATION },
    { "IOCTL_FSVIDEO_COPY_FRAME_BUFFER",                IOCTL_FSVIDEO_COPY_FRAME_BUFFER },
    { "IOCTL_FSVIDEO_WRITE_TO_FRAME_BUFFER",            IOCTL_FSVIDEO_WRITE_TO_FRAME_BUFFER },
    { "IOCTL_FSVIDEO_REVERSE_MOUSE_POINTER",            IOCTL_FSVIDEO_REVERSE_MOUSE_POINTER },
    { "IOCTL_FSVIDEO_SET_CURRENT_MODE",                 IOCTL_FSVIDEO_SET_CURRENT_MODE },
    { "IOCTL_FSVIDEO_SET_SCREEN_INFORMATION",           IOCTL_FSVIDEO_SET_SCREEN_INFORMATION },
    { "IOCTL_FSVIDEO_SET_CURSOR_POSITION",              IOCTL_FSVIDEO_SET_CURSOR_POSITION },
    { "UNKNOWN IOCTL",                                  0 },

};

#define NUM_IOCTLS (sizeof(allStrings)/sizeof(VIDEO_STRING))

char *FindIoctl (ULONG code)
{
    unsigned int i;

    for (i = 0; i < NUM_IOCTLS; i++)
    {
        if (allStrings[i].code == code)
            return allStrings[i].string;

    }

    return allStrings[NUM_IOCTLS - 1].string;
}
#endif