 /*
  * <LIC_AMD_STD>
  * Copyright (C) <years> Advanced Micro Devices, Inc.  All Rights Reserved.
  * Copyright (c) 1992-1996 Microsoft Corporation
  * </LIC_AMD_STD>
  * 
  * <CTL_AMD_STD>
  * </CTL_AMD_STD>
  * 
  * <DOC_AMD_STD>
  * Support routines to draw small strips inside DrvStrokePath.
  * </DOC_AMD_STD>
  * 
  */

#include "precomp.h"
#include "gfx_regs.h"
#include "gfx_defs.h"

#define GU2_WAIT_PENDING while(READ_GP32(MGP_BLT_STATUS) & MGP_BS_BLT_PENDING)
#define GU2_WAIT_BUSY while(READ_GP32(MGP_BLT_STATUS) & MGP_BS_BLT_BUSY)

extern unsigned char *gfx_virt_fbptr;
extern unsigned char *gfx_virt_gpptr;
extern unsigned long  gu2_bpp;
extern unsigned char  mode_shift;

/**************************************************************************
 * VOID stripSolidHorizontal
 *
 * Draws left-to-right x-major near-horizontal lines using radial lines.
 *
 **************************************************************************/

VOID
stripSolidHorizontal(
	PDEV*		ppdev,
    ULONG       dstOffset,
    ULONG       dstPitch,
    ULONG       VecMode,
	STRIP*		pStrip,
	LINESTATE*	pLineState)
{
	LONG	cStrips;
	LONG	i, yInc, x, y;
	PLONG	pStrips;
	ULONG   blitMode;

	DISPDBG((ppdev, 1, "stripSolidHorizontal\n"));

	cStrips = pStrip->cStrips;

	x = pStrip->ptlStart.x;
	y = pStrip->ptlStart.y;

	DISPDBG((ppdev, 1, "strips %d, start (%d,%d), flags %x\n",
		cStrips, x, y, pStrip->flFlips));

	// CHECK DESTINATION REQUIRED

    blitMode = 0;
    if (VecMode)
        blitMode = MGP_BM_DST_REQ;
	
	yInc = 1;
	if (pStrip->flFlips & FL_FLIP_V)
		yInc = -1;

	pStrips = pStrip->alStrips;

	for (i = 0; i < cStrips; i++)
	{
        GU2_WAIT_PENDING;

        WRITE_GP32 (MGP_STRIDE, dstPitch);
        WRITE_GP32 (MGP_DST_OFFSET, 
            (y * dstPitch) +
            (x << mode_shift) + dstOffset);
        WRITE_GP32 (MGP_WID_HEIGHT, ((*pStrips) << 16) | 1);
        WRITE_GP32 (MGP_BLT_MODE, blitMode);

        x += *pStrips++;
        y += yInc;
	}

	pStrip->ptlStart.x = x;
	pStrip->ptlStart.y = y;
}

/**************************************************************************
 * VOID stripSolidVertical
 *
 * Draws left-to-right y-major near-vertical lines using radial lines.
 *
 **************************************************************************/

VOID stripSolidVertical(
	PDEV*		ppdev,
    ULONG       dstOffset,
    ULONG       dstPitch,
    ULONG       VecMode,
	STRIP*		pStrip,
	LINESTATE*	pLineState)
{
	LONG	cStrips;
	LONG	i, x, y;
	PLONG	pStrips;
    ULONG   bltMode;

	DISPDBG((ppdev, 1, "stripSolidVertical\n"));

	cStrips = pStrip->cStrips;
	pStrips = pStrip->alStrips;

	x = pStrip->ptlStart.x;
	y = pStrip->ptlStart.y;

	DISPDBG((ppdev, 1, "strips %d, start (%d,%d), flags %x, dest %d\n",
		cStrips, x, y, pStrip->flFlips, ppdev->destMatters));

    bltMode = 0;
    if (VecMode)
        bltMode = MGP_BM_DST_REQ;

	if (!(pStrip->flFlips & FL_FLIP_V))
	{
		for (i = 0; i < cStrips; i++)
		{
            GU2_WAIT_PENDING;

            WRITE_GP32 (MGP_STRIDE, dstPitch);
            WRITE_GP32 (MGP_DST_OFFSET, 
                (y * dstPitch) +
                (x << mode_shift) + dstOffset);
            WRITE_GP32 (MGP_WID_HEIGHT, (1 << 16) | (*pStrips & 0xFFFF)); 
            WRITE_GP32 (MGP_BLT_MODE, bltMode);

			y += *pStrips++;
			x++;
		}
	}
	else
	{
		for (i = 0; i < cStrips; i++)
		{
            GU2_WAIT_PENDING;

            WRITE_GP32 (MGP_STRIDE, dstPitch);
            WRITE_GP32 (MGP_DST_OFFSET, 
                ((y - *pStrips + 1) * dstPitch) +
                (x << mode_shift) + 
                dstOffset);
            WRITE_GP32 (MGP_WID_HEIGHT, (1 << 16) | (*pStrips & 0xFFFF)); 
            WRITE_GP32 (MGP_BLT_MODE, bltMode);

			y -= *pStrips++;
			x++;
		}
	}

	pStrip->ptlStart.x = x;
	pStrip->ptlStart.y = y;
}

/***************************************************************************
 * VOID stripSolidDiagonalHorizontal
 *
 * Draws left-to-right x-major near-diagonal lines using radial lines.
 *
 **************************************************************************/

VOID stripSolidDiagonalHorizontal(
	PDEV*		ppdev,
    ULONG       dstOffset,
    ULONG       dstPitch,
    ULONG       VecMode,
	STRIP*		pStrip,
	LINESTATE*	pLineState)
{
	LONG	cStrips;
	LONG	i, x, y;
	PLONG	pStrips;
	ULONG	vectorMode;

	DISPDBG((ppdev, 1, "stripSolidDiagonalHorizontal\n"));

	cStrips = pStrip->cStrips;
	pStrips = pStrip->alStrips;

	x = pStrip->ptlStart.x;
	y = pStrip->ptlStart.y;

	DISPDBG((ppdev, 1, "strips %d, start (%d,%d), flags %x\n",
		cStrips, x, y, pStrip->flFlips));

	if (!(pStrip->flFlips & FL_FLIP_V))
	{
		for (i = 0; i < cStrips; i++)
		{
			DISPDBG((ppdev, 0, "x/y (%d,%d), width %d\n", x, y, *pStrips));

			vectorMode = VecMode | GP_VECTOR_MAJOR_AXIS_POS | GP_VECTOR_MINOR_AXIS_POS | GP_VECTOR_X_MAJOR;

            GU2_WAIT_PENDING;

            WRITE_GP32 (MGP_STRIDE, dstPitch);
            WRITE_GP32 (MGP_DST_OFFSET, 
                (y * dstPitch) +
                (x << mode_shift) + dstOffset);
	        WRITE_GP32 (MGP_VEC_LEN, (*pStrips << 16) | 1);
	        WRITE_GP32 (MGP_VEC_ERR, 0);
	        WRITE_GP32 (MGP_VECTOR_MODE, vectorMode);

			y += *pStrips - 1;
			x += *pStrips++;
		}
	}
	else
	{
		for (i = 0; i < cStrips; i++)
		{
			DISPDBG((ppdev, 0, "flip x/y (%d,%d), width %d\n", x, y, *pStrips));


			vectorMode = VecMode | GP_VECTOR_MAJOR_AXIS_POS | GP_VECTOR_MINOR_AXIS_NEG | GP_VECTOR_X_MAJOR;

            GU2_WAIT_PENDING;

            WRITE_GP32 (MGP_STRIDE, dstPitch);
            WRITE_GP32 (MGP_DST_OFFSET, 
                (y * dstPitch) +
                (x << mode_shift) + dstOffset);
	        WRITE_GP32 (MGP_VEC_LEN, (*pStrips << 16) | 1);
	        WRITE_GP32 (MGP_VEC_ERR, 0);
	        WRITE_GP32 (MGP_VECTOR_MODE, vectorMode);

			y -= *pStrips - 1;
			x += *pStrips++;
		}
	}

	pStrip->ptlStart.x = x;
	pStrip->ptlStart.y = y;
}

/**************************************************************************\ 
 * VOID stripSolidDiagonalVertical
 *
 * Draws left-to-right y-major near-diagonal lines using radial lines.
 *
 **************************************************************************/

VOID stripSolidDiagonalVertical(
	PDEV*		ppdev,
    ULONG       dstOffset,
    ULONG       dstPitch,
    ULONG       VecMode,
	STRIP*		pStrip,
	LINESTATE*	pLineState)
{
	LONG	cStrips;
	LONG	i, x, y;
	PLONG	pStrips;
	ULONG	vectorMode;

	DISPDBG((ppdev, 1, "stripSolidDiagonalVertical\n"));

	cStrips = pStrip->cStrips;
	pStrips = pStrip->alStrips;

	x = pStrip->ptlStart.x;
	y = pStrip->ptlStart.y;

	DISPDBG((ppdev, 1, "strips %d, start (%d,%d), flags %x\n",
		cStrips, x, y, pStrip->flFlips));

	if (!(pStrip->flFlips & FL_FLIP_V))
	{
		for (i = 0; i < cStrips; i++)
		{
			DISPDBG((ppdev, 0, "x/y (%d,%d), height %d\n", x, y, *pStrips));
			
            vectorMode = VecMode | GP_VECTOR_MAJOR_AXIS_POS | GP_VECTOR_MINOR_AXIS_POS | GP_VECTOR_X_MAJOR;
			
            GU2_WAIT_PENDING;

            WRITE_GP32 (MGP_STRIDE, dstPitch);
            WRITE_GP32 (MGP_DST_OFFSET, 
                (y * dstPitch) +
                (x << mode_shift) + dstOffset);
	        WRITE_GP32 (MGP_VEC_LEN, (*pStrips << 16) | 1);
	        WRITE_GP32 (MGP_VEC_ERR, 0);
	        WRITE_GP32 (MGP_VECTOR_MODE, vectorMode);

			y += *pStrips;
			x += *pStrips++ - 1;
		}
	}
	else
	{
		for (i = 0; i < cStrips; i++)
		{
			DISPDBG((ppdev, 0, "flip x/y (%d,%d), height %d\n", x, y, *pStrips));
			vectorMode = VecMode | GP_VECTOR_MAJOR_AXIS_POS | GP_VECTOR_MINOR_AXIS_NEG | GP_VECTOR_X_MAJOR;

            GU2_WAIT_PENDING;

            WRITE_GP32 (MGP_STRIDE, dstPitch);
            WRITE_GP32 (MGP_DST_OFFSET, 
                (y * dstPitch) +
                (x << mode_shift) + dstOffset);
	        WRITE_GP32 (MGP_VEC_LEN, (*pStrips << 16) | 1);
	        WRITE_GP32 (MGP_VEC_ERR, 0);
	        WRITE_GP32 (MGP_VECTOR_MODE, vectorMode);

			y -= *pStrips;
			x += *pStrips++ - 1;
		}
	}
	pStrip->ptlStart.x = x;
	pStrip->ptlStart.y = y;
}

/**************************************************************************
 * VOID vStripStyledHorizontal
 *
 * Takes the list of strips that define the pixels that would be lit for
 * a solid line, and breaks them into styling chunks according to the
 * styling information that is passed in.
 *
 * This particular routine handles x-major lines that run left-to-right,
 * and are comprised of horizontal strips.  It draws the dashes using
 * short-stroke vectors.
 *
 * The performance of this routine could be improved significantly if
 * anyone cared enough about styled lines improve it.
 *
 **************************************************************************/

VOID vStripStyledHorizontal(
	PDEV*		ppdev,
    ULONG       dstOffset,
    ULONG       dstPitch,
    ULONG       VecMode,
	STRIP*		pstrip,
	LINESTATE*	pls)
{
	LONG	x;
	LONG	y;
	ULONG	dirSkip;
	LONG	dy;
	LONG*	plStrip;
	LONG	cStrips;
	LONG	cStyle;
	LONG	cStrip;
	LONG	cThis;
	ULONG	bIsGap;
	ULONG   bltMode;

	DISPDBG((ppdev, 1, "vStripStyledHorizontal\n"));

	if (pstrip->flFlips & FL_FLIP_V)
	{
		// The minor direction of the line is 90 degrees, and the major
		// direction is 0 (it's a left-to-right x-major line going up):
		dy		= -1;
	}
	else
	{
		// The minor direction of the line is 270 degrees, and the major
		// direction is 0 (it's a left-to-right x-major line going down):
		dy		= 1;
	}


    if (VecMode) bltMode = MGP_BM_DST_REQ;
    else         bltMode = 0;
	
	cStrips = pstrip->cStrips;		// Total number of strips we'll do
	plStrip = pstrip->alStrips; 	// Points to current strip
	x		= pstrip->ptlStart.x;	// x position of start of first strip
	y		= pstrip->ptlStart.y;	// y position of start of first strip

	DISPDBG((ppdev, 0, "strips %d, start %d, %d\n", cStrips, x, y));

	cStrip = *plStrip;				// Number of pels in first strip

	cStyle = pls->spRemaining;		// Number of pels in first 'gap' or 'dash'
	bIsGap = pls->ulStyleMask;		// Tells whether in a 'gap' or a 'dash'

	// ulStyleMask is non-zero if we're in the middle of a 'gap',
	// and zero if we're in the middle of a 'dash':

	if (bIsGap)
		goto SkipAGap;
	else
		goto OutputADash;

PrepareToSkipAGap:
	DISPDBG((ppdev, 0, "Pskip gap\n"));

	// Advance in the style-state array, so that we can find the next
	// 'dot' that we'll have to display:

	bIsGap = ~bIsGap;
	pls->psp++;
	if (pls->psp > pls->pspEnd)
		pls->psp = pls->pspStart;

	cStyle = *pls->psp;

	// If 'cStrip' is zero, we also need a new strip:

	if (cStrip != 0)
		goto SkipAGap;

	// Here, we're in the middle of a 'gap' where we don't have to
	// display anything.  We simply cycle through all the strips
	// we can, keeping track of the current position, until we run
	// out of 'gap':

	while (TRUE)
	{
		// Each time we loop, we move to a new scan and need a new strip:

		y += dy;

		plStrip++;
		cStrips--;
		if (cStrips == 0)
			goto AllDone;

		cStrip = *plStrip;

	SkipAGap:
		DISPDBG((ppdev, 0, "skip gap\n"));

		cThis	= min(cStrip, cStyle);
		cStyle -= cThis;
		cStrip -= cThis;

		x += cThis;

		if (cStyle == 0)
			goto PrepareToOutputADash;
	}

PrepareToOutputADash:
	DISPDBG((ppdev, 0, "Poutput dash\n"));

	// Advance in the style-state array, so that we can find the next
	// 'dot' that we'll have to display:

	bIsGap = ~bIsGap;
	pls->psp++;
	if (pls->psp > pls->pspEnd)
		pls->psp = pls->pspStart;

	cStyle = *pls->psp;

	// If 'cStrip' is zero, we also need a new strip.

	if (cStrip != 0)
	{
		goto OutputADash;
	}

	while (TRUE)
	{
		// Each time we loop, we move to a new scan and need a new strip:

		y += dy;

		plStrip++;
		cStrips--;
		if (cStrips == 0)
			goto AllDone;

		cStrip = *plStrip;

	OutputADash:
		DISPDBG((ppdev, 0, "output dash\n"));

		cThis	= min(cStrip, cStyle);
		cStyle -= cThis;
		cStrip -= cThis;

        GU2_WAIT_PENDING;

        WRITE_GP32 (MGP_STRIDE, dstPitch);
        WRITE_GP32 (MGP_DST_OFFSET, 
            (y * dstPitch) +
            (x << mode_shift) + dstOffset);
        WRITE_GP32 (MGP_WID_HEIGHT, (cThis << 16) | 1);
        WRITE_GP32 (MGP_BLT_MODE, bltMode);

		x += cThis;

		if (cStyle == 0)
			goto PrepareToSkipAGap;
	}

AllDone:

	// Update our state variables so that the next line can continue
	// where we left off:

	pls->spRemaining   = cStyle;
	pls->ulStyleMask   = bIsGap;
	pstrip->ptlStart.x = x;
	pstrip->ptlStart.y = y;
}

/**************************************************************************
 * VOID vStripStyledVertical
 *
 * Takes the list of strips that define the pixels that would be lit for
 * a solid line, and breaks them into styling chunks according to the
 * styling information that is passed in.
 *
 * This particular routine handles y-major lines that run left-to-right,
 * and are comprised of vertical strips.  It draws the dashes using
 * short-stroke vectors.
 *
 * The performance of this routine could be improved significantly if
 * anyone cared enough about styled lines improve it.
 *
 **************************************************************************/

VOID vStripStyledVertical(
	PDEV*		ppdev,
	ULONG       dstOffset,
    ULONG       dstPitch,
    ULONG       VecMode,
    STRIP*		pstrip,
	LINESTATE*	pls)
{
	LONG	x;
	LONG	y;
	ULONG	dirSkip;
	ULONG	dirSkip16;
	LONG	dy;
	LONG*	plStrip;
	LONG	cStrips;
	LONG	cStyle;
	LONG	cStrip;
	LONG	cThis;
	ULONG	bIsGap;
	ULONG   bltMode;

	DISPDBG((ppdev, 1, "vStripStyledVertical\n"));

	if (pstrip->flFlips & FL_FLIP_V)
	{
		// The minor direction of the line is 0 degrees, and the major
		// direction is 90 (it's a left-to-right y-major line going up):

		dirSkip   = 0x0150;
		dirSkip16 = 0x5f51; 		// For drawing 16 pels straight up
		dy		  = -1;
	}
	else
	{
		// The minor direction of the line is 0 degrees, and the major
		// direction is 270 (it's a left-to-right y-major line going down):

		dirSkip   = 0x01d0;
		dirSkip16 = 0xdfd1; 		// For drawing 16 pels straight down
		dy		  = 1;
	}

    bltMode = 0;
    if (VecMode)
        bltMode = MGP_BM_DST_REQ;

	cStrips = pstrip->cStrips;		// Total number of strips we'll do
	plStrip = pstrip->alStrips; 	// Points to current strip
	x		= pstrip->ptlStart.x;	// x position of start of first strip
	y		= pstrip->ptlStart.y;	// y position of start of first strip

	cStrip = *plStrip;				// Number of pels in first strip

	cStyle = pls->spRemaining;		// Number of pels in first 'gap' or 'dash'
	bIsGap = pls->ulStyleMask;		// Tells whether in a 'gap' or a 'dash'

	// ulStyleMask is non-zero if we're in the middle of a 'gap',
	// and zero if we're in the middle of a 'dash':

	if (bIsGap)
		goto SkipAGap;
	else
		goto OutputADash;

PrepareToSkipAGap:

	// Advance in the style-state array, so that we can find the next
	// 'dot' that we'll have to display:

	bIsGap = ~bIsGap;
	pls->psp++;
	if (pls->psp > pls->pspEnd)
		pls->psp = pls->pspStart;

	cStyle = *pls->psp;

	// If 'cStrip' is zero, we also need a new strip:

	if (cStrip != 0)
		goto SkipAGap;

	// Here, we're in the middle of a 'gap' where we don't have to
	// display anything.  We simply cycle through all the strips
	// we can, keeping track of the current position, until we run
	// out of 'gap':

	while (TRUE)
	{
		// Each time we loop, we move to a new column and need a new strip:

		x++;

		plStrip++;
		cStrips--;
		if (cStrips == 0)
			goto AllDone;

		cStrip = *plStrip;

	SkipAGap:

		cThis	= min(cStrip, cStyle);
		cStyle -= cThis;
		cStrip -= cThis;

		y += (dy > 0) ? cThis : -cThis;

		if (cStyle == 0)
			goto PrepareToOutputADash;
	}

PrepareToOutputADash:

	// Advance in the style-state array, so that we can find the next
	// 'dot' that we'll have to display:

	bIsGap = ~bIsGap;
	pls->psp++;
	if (pls->psp > pls->pspEnd)
		pls->psp = pls->pspStart;

	cStyle = *pls->psp;

	// If 'cStrip' is zero, we also need a new strip.

	if (cStrip != 0)
	{
		goto OutputADash;
	}

	while (TRUE)
	{
		// Each time we loop, we move to a new column and need a new strip:

		x++;

		plStrip++;
		cStrips--;
		if (cStrips == 0)
			goto AllDone;

		cStrip = *plStrip;

	OutputADash:

		cThis	= min(cStrip, cStyle);
		cStyle -= cThis;
		cStrip -= cThis;

        GU2_WAIT_PENDING;

        WRITE_GP32 (MGP_STRIDE, dstPitch);
        if (dy < 0)
        {
            WRITE_GP32 (MGP_DST_OFFSET, 
                ((y - cThis + 1) * dstPitch) +
                (x << mode_shift) + dstOffset);
        }
        else
        {
            WRITE_GP32 (MGP_DST_OFFSET, 
                (y * dstPitch) +
                (x << mode_shift) + dstOffset);
        }
        WRITE_GP32 (MGP_WID_HEIGHT, (1 << 16) | cThis);
        WRITE_GP32 (MGP_BLT_MODE, bltMode);

		y += (dy > 0) ? cThis : -cThis;

		if (cStyle == 0)
			goto PrepareToSkipAGap;
	}

AllDone:

	// Update our state variables so that the next line can continue
	// where we left off:

	pls->spRemaining   = cStyle;
	pls->ulStyleMask   = bIsGap;
	pstrip->ptlStart.x = x;
	pstrip->ptlStart.y = y;
}
