 /*
  * <LIC_AMD_STD>
  * Copyright (C) <years> Advanced Micro Devices, Inc.  All Rights Reserved.
  * </LIC_AMD_STD>
  * 
  * <CTL_AMD_STD>
  * </CTL_AMD_STD>
  * 
  * <DOC_AMD_STD>
  * Hardware Cursor Support.
  * </DOC_AMD_STD>
  * 
  */

#include "precomp.h"

/////////////////////////////////////////////////////////////////////////
// GXSetCursorPos
//
// Private routine to adjust the current pointer position.  If panning
// occurs and a video window is active, the current video position is also 
// updated.
//
void GXSetCursorPos(PDEV *ppdev, short x, short y)
{
	ULONG cursor_offset, offset;
	static panOffset = 0;

	DISPDBG((ppdev,0, "GXSetCursorPos in Display\n"));
	
	cursor_offset = ppdev->myfuncs.CursorOffset;

	ppdev->myfuncs.pfn_set_cursor_position(cursor_offset, x, y, ppdev->xHotSpot, ppdev->yHotSpot);
	
	if (ppdev->OverlayActive && *(ppdev->myfuncs.pPanelEnable))
	{
		offset = ppdev->myfuncs.pfn_get_display_offset();

		if (offset != panOffset)
		{
			SetVideoPosition (ppdev, ppdev->video_x, ppdev->video_y, 0, 0, FALSE);
			panOffset = offset;
		}
	}
}

/////////////////////////////////////////////////////////////////////////
// DrvMovePointer
//
// Moves the hardware pointer to a new position.
//
VOID DrvMovePointer(
	SURFOBJ * pso,
	LONG x,
	LONG y,
	RECTL * prcl)
{
	PDEV *ppdev = (PDEV *) pso->dhpdev;
	
	// CHECK IF CURSOR SHOULD BE DISABLED 
	// A new x position of -1 means hide the pointer. 
    //
	if (x == -1) 
	{
		ppdev->myfuncs.pfn_set_cursor_enable (0);
		ppdev->cursorEnabled = FALSE;
	} 
	else 
	{	
		// CHANGE POINTER POSITION 
        //
		if (y < 0)
        {
            // When y is less than 0, this is a callback enabling us to pan
            // a display with a simulated cursor.  If the cursor is not 
            // simulated, we will have already received a normal call to
            // DrvMovePointer by this point.
            //
            if (ppdev->bSimulatedCursor || !ppdev->cursorEnabled)
			    GXSetCursorPos (ppdev, (short)x, (short)(y + ppdev->cyScreen));
        }
		else
		{
			if (!ppdev->cursorEnabled)
			{
				ppdev->cursorEnabled = TRUE;
				ppdev->myfuncs.pfn_set_cursor_enable (1);
			}
			GXSetCursorPos (ppdev, (short)x, (short)y);
		}
	}
}

/////////////////////////////////////////////////////////////////////////
// bCopyMonoPointer
//
// Copies two monochrome masks into a buffer of the maximum size handled by the
// hardware, with any extra bits set to 0.  The masks are converted to topdown
// form if they aren't already.  Returns TRUE if we can handle this pointer in
// hardware, FALSE if not.
//
BOOL bCopyMonoPointer(
	PDEV *ppdev,
	SURFOBJ * pso)
{
	ULONG cxSrc, cySrc;
	ULONG x, y, index;
	ULONG srcBytes, maxBytes;
	BYTE andArray[512], xorArray[512];
	PBYTE pSrcAnd, pSrcXor;
    PBYTE pDstAnd, pDstXor;

	// GRAB CURSOR SIZES                 
	// Cursor height includes both masks 
    //
 	cxSrc = pso->sizlBitmap.cx;
	cySrc = pso->sizlBitmap.cy >> 1;

	// CHECK SIZE 
	// Make sure the new pointer isn't too big to handle 
	//
	if ((cxSrc > MAX_GX2_CURSOR_WIDTH) ||
		(cySrc > MAX_GX2_CURSOR_HEIGHT)) 
	{
		return FALSE;
	}

	// FILL IN CUSTOM CURSOR ARRAY 
	// We are going to call Durango directly from here.  This simplifies 
	// the interface, as Durango expects an array of XOR and AND data.   
	// Unfortunately, however, the arrays are expected to be constant    
	// size and the expected byte ordering is reversed.                  
    //
	srcBytes   = (cxSrc + 7) >> 3;
	maxBytes   = (MAX_GX2_CURSOR_WIDTH + 7) >> 3;
	pSrcAnd    = (PBYTE) pso->pvBits;
	pSrcXor    = pSrcAnd + (cySrc * pso->lDelta);
	pDstAnd    = &andArray[0];
    pDstXor    = &xorArray[0];

	for (y = 0; y < cySrc; y++)
	{
		// FILL IN VALID BYTES 
        //
		for (x = 0; x < srcBytes; x++)
		{
            index = (x & 0xFFC) + 3 - (x & 3);
            pDstAnd[index] = pSrcAnd[x];
            pDstXor[index] = pSrcXor[x];
		}

		// FILL IN REMAINING BYTES WITH TRANSPARENCY 
        //
        for (x = srcBytes; x < maxBytes; x++)
        {
            index = (x & 0xFFC) + 3 - (x & 3);
            pDstAnd[index] = 0xFF;
            pDstXor[index] = 0x00;
        }
		
		// INCREMENT TO NEXT LINE OF DATA 
        //
		pSrcAnd += pso->lDelta;
		pSrcXor += pso->lDelta;
        pDstAnd += 8;
        pDstXor += 8;
	}

	// PAD REMAINING LINES WITH TRANSPARENCY 
    //
	for (y = cySrc; y < MAX_GX2_CURSOR_HEIGHT; y++)
	{
        *((ULONG *)(pDstAnd))     = 0xFFFFFFFF;
        *((ULONG *)(pDstAnd + 4)) = 0xFFFFFFFF;
        *((ULONG *)(pDstXor))     = 0x00000000;
        *((ULONG *)(pDstXor + 4)) = 0x00000000;
		
        pDstAnd += 8;
        pDstXor += 8;
	}
		
	// CALL DURANGO TO INITIALIZE CURSOR 
    //
	ppdev->myfuncs.pfn_set_cursor_shape64 (ppdev->myfuncs.CursorOffset, (ULONG *)(&andArray[0]), (ULONG *)(&xorArray[0]));

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////
// bSetHardwarePointerShape
//
// Changes the shape of the Hardware Pointer.
//
// Returns: True if successful, False if Pointer shape can't be hardware.
//
BOOL bSetHardwarePointerShape(   
	SURFOBJ *pso,
	SURFOBJ *psoMask,
	SURFOBJ *psoColor,
	XLATEOBJ *pxlo,
	LONG x,
	LONG y,
	FLONG fl)
{
	PDEV *ppdev = (PDEV *) pso->dhpdev;
	DWORD returnedDataLength;

	DISPDBG((ppdev, 0, "bSetHardwarePointerShape: file %s, line %d\n", __FILE__, __LINE__));

	// CHECK COLOR CURSOR (UNSUPPORTED)                                                             
    //
	if (psoColor != (SURFOBJ *) NULL) 
	{
		return FALSE;
	} 
	else 
	{
		// MONOCHROME CURSOR 
        //
		if (!bCopyMonoPointer(ppdev, psoMask)) 
		{
            return FALSE;
		} 
		ppdev->myfuncs.pfn_set_cursor_colors (0x00000000, 0x00FCFEFD);
	}

	// INITIALIZE POINTER ATTRIBUTES AND POSITION                        
	// We will call Durango directly, instead of using an IOCTL service. 
    //
	if (x != -1 || y != -1)
	{
        ppdev->myfuncs.pfn_set_cursor_position (ppdev->myfuncs.CursorOffset, (short)x, (short)y, 
            ppdev->xHotSpot, ppdev->yHotSpot);
		ppdev->myfuncs.pfn_set_cursor_enable (1);
		ppdev->cursorEnabled = TRUE;
	}
	
	return (TRUE);
}

/////////////////////////////////////////////////////////////////////////
// DrvSetPointerShape
//
// Sets the new pointer shape.
//
ULONG DrvSetPointerShape(
	SURFOBJ * pso,
	SURFOBJ * psoMask,
	SURFOBJ * psoColor,
	XLATEOBJ * pxlo,
	LONG xHot,
	LONG yHot,
	LONG x,
	LONG y,
	RECTL * prcl,
	FLONG fl)
{
	PDEV *ppdev = (PDEV *) pso->dhpdev;
	DWORD returnedDataLength;

	// STORE HOTSPOTS 
    //
	ppdev->xHotSpot = (short)xHot;
	ppdev->yHotSpot = (short)yHot;

	// PROGRAM CURSOR 
    //
	if (!bSetHardwarePointerShape(pso, psoMask, psoColor, pxlo, x, y, fl)) 
	{
		// UNSUPPORTED CURSOR - DISABLE HARDWARE CURSOR 

		if (ppdev->cursorEnabled) 
		{
			ppdev->cursorEnabled = FALSE;
			ppdev->myfuncs.pfn_set_cursor_enable(0);
		}

		// MINIPORT DECLINES TO REALIZE THIS POINTER 

        ppdev->bSimulatedCursor = TRUE;
		return SPS_DECLINE;
	} 

    ppdev->bSimulatedCursor = FALSE;

	// SET NOEXCLUDE 
	// Noexclude informs Windows that the cursor is independent of all 
	// drawing operations.                                             
    //
	return SPS_ACCEPT_NOEXCLUDE;
}

