 /*
  * <LIC_AMD_STD>
  * Copyright (C) <years> Advanced Micro Devices, Inc.  All Rights Reserved.
  * Copyright (c) 1992-1995 Microsoft Corporation
  * </LIC_AMD_STD>
  * 
  * <CTL_AMD_STD>
  * </CTL_AMD_STD>
  * 
  * <DOC_AMD_STD>
  * Display Palette support.
  * </DOC_AMD_STD>
  * 
  */

#include "precomp.h"

/* Global Table defining the 20 Window default colours.  For 256 colour
 * palettes the first 10 must be put at the beginning of the palette
 * and the last 10 at the end of the palette.
 */
PALETTEENTRY gapalBase[20] =
{
	{ 0,   0,	0,	 0 },		// 0
	{ 0x80,0,	0,	 0 },		// 1
	{ 0,   0x80,0,	 0 },		// 2
	{ 0x80,0x80,0,	 0 },		// 3
	{ 0,   0,	0x80,0 },		// 4
	{ 0x80,0,	0x80,0 },		// 5
	{ 0,   0x80,0x80,0 },		// 6
	{ 0xC0,0xC0,0xC0,0 },		// 7
	{ 192, 220, 192, 0 },		// 8
	{ 166, 202, 240, 0 },		// 9
	{ 255, 251, 240, 0 },		// 10
	{ 160, 160, 164, 0 },		// 11
	{ 0x80,0x80,0x80,0 },		// 12
	{ 0xFF,0,	0	,0 },		// 13
	{ 0,   0xFF,0	,0 },		// 14
	{ 0xFF,0xFF,0	,0 },		// 15
	{ 0   ,0,	0xFF,0 },		// 16
	{ 0xFF,0,	0xFF,0 },		// 17
	{ 0,   0xFF,0xFF,0 },		// 18
	{ 0xFF,0xFF,0xFF,0 },		// 19
};

/******************************Public*Routine******************************\
* BOOL bInitializePalette
*
* Initializes default palette for PDEV.
*
\**************************************************************************/

BOOL
bInitializePalette(
	PDEV*	 ppdev,
	DEVINFO* pdi)
{
	PALETTEENTRY*	ppal;
	PALETTEENTRY*	ppalTmp;
	ULONG			ulLoop;
	BYTE			jRed;
	BYTE			jGre;
	BYTE			jBlu;
	HPALETTE		hpal;

	DISPDBG((ppdev, 200, "bInitPalette format %x (8bpp = %x)\n",
			ppdev->iBitmapFormat, BMF_8BPP));

	if (ppdev->iBitmapFormat == BMF_8BPP)
	{
		// Allocate our palette:

		ppal = (PALETTEENTRY*)EngAllocMem(FL_ZERO_MEMORY,
						(sizeof(PALETTEENTRY) * 256), gAllocTag);
		if (ppal == NULL)
			goto ReturnFalse;

		ppdev->pPal = ppal;

		// Generate 256 (8*4*4) RGB combinations to fill the palette

		jRed = 0;
		jGre = 0;
		jBlu = 0;

		ppalTmp = ppal;

		for (ulLoop = 256; ulLoop != 0; ulLoop--)
		{
			ppalTmp->peRed	 = jRed;
			ppalTmp->peGreen = jGre;
			ppalTmp->peBlue  = jBlu;
			ppalTmp->peFlags = 0;

			ppalTmp++;

			if (!(jRed += 32))
				if (!(jGre += 32))
					jBlu += 64;
		}

		// Fill in Windows reserved colours from the WIN 3.0 DDK
		// The Window Manager reserved the first and last 10 colours for
		// painting windows borders and for non-palette managed applications.

		for (ulLoop = 0; ulLoop < 10; ulLoop++)
		{
			// First 10

			ppal[ulLoop]	   = gapalBase[ulLoop];

			// Last 10

			ppal[246 + ulLoop] = gapalBase[ulLoop+10];
		}

		// Create handle for palette.

		hpal = EngCreatePalette(PAL_INDEXED, 256, (ULONG*) ppal, 0, 0, 0);
	}
	else
	{
		ASSERTVDD((ppdev->iBitmapFormat == BMF_16BPP) ||
				 (ppdev->iBitmapFormat == BMF_32BPP),
				 "This case handles only 16 or 32bpp");

		hpal = EngCreatePalette(PAL_BITFIELDS, 0, NULL,
								ppdev->flRed, ppdev->flGreen, ppdev->flBlue);
	}

	ppdev->hpalDefault = hpal;
	pdi->hpalDefault   = hpal;

	if (hpal == 0)
		goto ReturnFalse;

	return(TRUE);

ReturnFalse:

	DISPDBG((ppdev, 0, "Failed bInitializePalette"));
	return(FALSE);
}

/******************************Public*Routine******************************\
* VOID vUninitializePalette
*
* Frees resources allocated by bInitializePalette.
*
* Note: In an error case, this may be called before bInitializePalette.
*
\**************************************************************************/

VOID
vUninitializePalette(PDEV* ppdev)
{
	DISPDBG((ppdev, 200, "vUnititializePalette\n"));

	// Delete the default palette if we created one:

	if (ppdev->hpalDefault != 0)
		EngDeletePalette(ppdev->hpalDefault);

	if (ppdev->pPal != (PALETTEENTRY*) NULL)
		EngFreeMem(ppdev->pPal);
}

/******************************Public*Routine******************************\
* BOOL bEnablePalette
*
* Initialize the hardware's 8bpp palette registers.
*
\**************************************************************************/

BOOL
bEnablePalette(PDEV* ppdev)
{
	BYTE		ajClutSpace[MAX_CLUT_SIZE];
	PVIDEO_CLUT pScreenClut;
	ULONG		ulReturnedDataLength;
	ULONG		cColors;
	PVIDEO_CLUTDATA pScreenClutData;

	DISPDBG((ppdev, 200, "bEnablePalette format %x (8bpp = %x)\n",
			ppdev->iBitmapFormat, BMF_8BPP));

	if (ppdev->iBitmapFormat == BMF_8BPP)
	{
		// Fill in pScreenClut header info:

		pScreenClut 			= (PVIDEO_CLUT) ajClutSpace;
		pScreenClut->NumEntries = 256;
		pScreenClut->FirstEntry = 0;

		// Copy colours in:

		cColors = 256;
		pScreenClutData = (PVIDEO_CLUTDATA) (&(pScreenClut->LookupTable[0]));

		while(cColors--)
		{
			pScreenClutData[cColors].Red =	  ppdev->pPal[cColors].peRed >>
											  ppdev->cPaletteShift;
			pScreenClutData[cColors].Green =  ppdev->pPal[cColors].peGreen >>
											  ppdev->cPaletteShift;
			pScreenClutData[cColors].Blue =   ppdev->pPal[cColors].peBlue >>
											  ppdev->cPaletteShift;
			pScreenClutData[cColors].Unused = 0;
		}

		// Set palette registers:

		if (EngDeviceIoControl(ppdev->hDriver,
							 IOCTL_VIDEO_SET_COLOR_REGISTERS,
							 pScreenClut,
							 MAX_CLUT_SIZE,
							 NULL,
							 0,
							 &ulReturnedDataLength))
		{
			DISPDBG((ppdev, 300, "Failed bEnablePalette"));
			return(FALSE);
		}
	}

	DISPDBG((ppdev, 5, "Passed bEnablePalette"));

	return(TRUE);
}

/******************************Public*Routine******************************\
* VOID vDisablePalette
*
* Undoes anything done in bEnablePalette.
*
\**************************************************************************/

VOID
vDisablePalette(
	PDEV*	ppdev)
{
	// Nothin' to do
}

/******************************Public*Routine******************************\
* BOOL DrvSetPalette
*
* DDI entry point for manipulating the palette.
*
\**************************************************************************/

BOOL
DrvSetPalette(
	DHPDEV	dhpdev,
	PALOBJ* ppalo,
	FLONG	fl,
	ULONG	iStart,
	ULONG	cColors)
{
	BYTE			ajClutSpace[MAX_CLUT_SIZE];
	PVIDEO_CLUT 	pScreenClut;
	PVIDEO_CLUTDATA pScreenClutData;
	PDEV*			ppdev;

	ppdev = (PDEV*) dhpdev;

	DISPDBG ((ppdev, 1, "DrvSetPalette\n"));
	
	// Fill in pScreenClut header info:

	pScreenClut 			= (PVIDEO_CLUT) ajClutSpace;
	pScreenClut->NumEntries = (USHORT) cColors;
	pScreenClut->FirstEntry = (USHORT) iStart;

	pScreenClutData = (PVIDEO_CLUTDATA) (&(pScreenClut->LookupTable[0]));

	if (cColors != PALOBJ_cGetColors(ppalo, iStart, cColors,
									 (ULONG*) pScreenClutData))
	{
		DISPDBG((ppdev, 0, "DrvSetPalette failed PALOBJ_cGetColors\n"));
		return FALSE;
	}

	// Set the high reserved byte in each palette entry to 0.
	// Do the appropriate palette shifting to fit in the DAC.

	if (ppdev->cPaletteShift)
	{
		while(cColors--)
		{
			pScreenClutData[cColors].Red >>= ppdev->cPaletteShift;
			pScreenClutData[cColors].Green >>= ppdev->cPaletteShift;
			pScreenClutData[cColors].Blue >>= ppdev->cPaletteShift;
			pScreenClutData[cColors].Unused = 0;
		}
	}
	else
	{
		while(cColors--)
		{
			pScreenClutData[cColors].Unused = 0;
		}
	}

	// Set palette registers

	if (EngDeviceIoControl(ppdev->hDriver,
						 IOCTL_VIDEO_SET_COLOR_REGISTERS,
						 pScreenClut,
						 MAX_CLUT_SIZE,
						 NULL,
						 0,
						 &cColors))
	{
		DISPDBG((ppdev, 0, "DrvSetPalette failed EngDeviceIoControl\n"));
		return FALSE;
	}

	return TRUE;

}

/*-------------------------------------------------------------------------
 * DrvIcmSetDeviceGammaRamp
 *
 * This routine programs the gamma RAM for the primary surface.
 -------------------------------------------------------------------------*/

BOOL DrvIcmSetDeviceGammaRamp (
    DHPDEV  dhpdev,
    ULONG   iFormat,
    LPVOID  lpRamp)
{
    GAMMARAMP *pGamma = (GAMMARAMP *)lpRamp;
    PDEV *ppdev = (PDEV *)dhpdev;
    unsigned long red, green, blue;
    int i, identity = 1;

    DISPDBG ((ppdev, 3000, "DrvIcmSetDeviceGammaRamp\n"));

    if (iFormat != IGRF_RGB_256WORDS)
    {
        DISPDBG ((ppdev, 3000, "Invalid Format for gamma ramp %i\n", iFormat));
        return FALSE;
    }

    if (ppdev->cPelSize == 0)
    {
        DISPDBG ((ppdev, 3000, "Unable to set gamma ramp in 8-bit modes.\n"));
        return FALSE;
    }

    /* CHECK IF WE ARE SETTING AN IDENTITY PALETTE */

    for (i = 0; i < 256; i++)
    {
        if ((pGamma->Red[i] >> 8) != i) 
        {
            identity = 0;
            break;
        }
        if ((pGamma->Green[i] >> 8) != i) 
        {
            identity = 0;
            break;
        }
        if ((pGamma->Blue[i] >> 8) != i) 
        {
            identity = 0;
            break;
        }
    }

    if (identity)
    {
        /* RESTORE VIDEO PALETTE AND DISPLAY PALETTE */

        ppdev->myfuncs.pfn_set_video_palette (NULL);
        for (i = 0; i < 256; i++)
        {
            ppdev->myfuncs.pfn_set_display_palette_entry (i, 
                (i << 16) | (i << 8) | i);
        }
    }
    else
    {
        for (i = 0; i < 256; i++)
        {
            red   = (ULONG)pGamma->Red[i]   >> 8;
            green = (ULONG)pGamma->Green[i] >> 8;
            blue  = (ULONG)pGamma->Blue[i]  >> 8;

            if (ppdev->cPelSize == 1)
            {
                ppdev->myfuncs.pfn_set_graphics_palette_entry (i, 
                    (red << 16) | (green << 8) | blue);
            }
            else
            {
                ppdev->myfuncs.pfn_set_display_palette_entry (i, 
                    (red << 16) | (green << 8) | blue);
            }
        }
    }

    return TRUE;
}