 /*
  * <LIC_AMD_STD>
  * Copyright (C) <years> Advanced Micro Devices, Inc.  All Rights Reserved.
  * </LIC_AMD_STD>
  * 
  * <CTL_AMD_STD>
  * </CTL_AMD_STD>
  * 
  * <DOC_AMD_STD>
  * Video Memory Heap Management routines.
  * </DOC_AMD_STD>
  * 
  */

#include "precomp.h"
#include "gfx_regs.h"
#include "gfx_defs.h"

#define GU2_WAIT_PENDING while(READ_GP32(MGP_BLT_STATUS) & MGP_BS_BLT_PENDING)

extern unsigned char *gfx_virt_fbptr;
extern unsigned char *gfx_virt_gpptr;
extern unsigned char *gfx_virt_vidptr;
extern unsigned char *gfx_virt_regptr;
extern unsigned long  gu2_bpp;
extern unsigned long mode_shift;
extern unsigned long mode_pitch;

// HEAP VARIABLES 
// The heap size is initialized in enable.c.                           
// The heap is maintained as an array of pointers.  The DHSURF value   
// is nothing more than an index into our array.  As we would like to be
// able to freely repack the heap whenever we choose, we maintain a separate
// array that holds the actual surface information.  If we decide to repack,
// we repack the data and adjust the main array.  We then adjust the members 
// in the dhsurf_array to point to the newly relocated bitmaps.  The actual GDI 
// SURFOBJ structures are never touched.  The only downside to this approach
// is that any locked surfaces that were created to allow punts become invalid 
// and must be deleted when the heap is compacted.
//
// The plot thickens.  DirectDraw surfaces must have priority over GDI surfaces.
// This is a WHQL requirement and is specified clearly in the DDK.  To eject a 
// surface from our heap, we must allocate space in memory for the data.  If we
// simply call EngModifySurface and clear the dhsurf, we will never be called
// to free the memory, resulting in a serious memory leak.  For now, our strategy
// is thus to allocate the space for the bitmaps, but not to change the dhsurf value.
// This implies that we will still be notified to delete the bitmap.  What's more,
// we can restore the bitmaps to the heap as soon as the DirectDraw heap is reset.
// Unfortunately, this also implies that we will be called to render to system
// memory bitmaps.
//
// SOME MORE WORDS ABOUT WINDOWS MEMORY MANAGEMENT... 
// This driver is written based on assumptions that were formed by observing 
// the behavior of the OS when it changes the display mode.  When the mode   
// is changed, Windows copies the contents of all bitmaps to DIBs.  However, 
// this occurs after the mode has been set.  We thus cannot clear the screen 
// or enable compression in the new mode as this may overwrite the bitmap    
// data before it is copied out.  We also cannot simply modify the surfaces  
// directly using EngModifySurface, as the bitmap handles maintained by the  
// have already been invalidated at that point.                              

#define HEAP_MAX 800

unsigned long heap_index     = 2;
unsigned long ddraw_index    = HEAP_MAX - 1;
unsigned long ddraw_max      = HEAP_MAX - 1;
unsigned long invalid_count  = 0;
unsigned long ddraw_invalid  = 0;
unsigned long ddraw_count    = 0;
unsigned long mode_number    = 0;
unsigned long mode_value     = 0;
unsigned long system_index   = 0;
unsigned long system_invalid = 0;

BITMAP_CACHE_ENTRY bitmap_heap[HEAP_MAX];
BITMAP_CACHE_ENTRY system_heap[HEAP_MAX];
unsigned long dhsurf_array[HEAP_MAX];
unsigned long dhsurf_start = 2;

/////////////////////////////////////////////////////////////////////////
// DrvCreateDeviceBitmap
//
// Routine responsible for allocation device-managed surfaces.
//
HBITMAP DrvCreateDeviceBitmap (
    DHPDEV  dhpdev,
    SIZEL   sizl,
    ULONG   iFormat)
{
	PDEV *ppdev = (PDEV *)dhpdev;
	unsigned long bitmap_pitch;
	unsigned long bitmap_size;
    unsigned long index;
	HBITMAP device_bitmap;
	
	if ((sizl.cx <= 8) && (sizl.cy <= 8))
		return 0;

    if (sizl.cx > 2048)
        return 0;

	if (!heap_start)
		return 0;

	if (old_mode_count)
		return 0;

	// CHECK FOR BITMAP IN AN UNSUPPORTED MODE                                
	// Monochrome bitmaps can be requested if the user calls CreateCompatibleBitmap
    // with a memory context that has not had a bitmap selected into it.  No other formats
    // should be possible.  However, Windows can be a little inconsistent...
    //
	if (iFormat != ppdev->iBitmapFormat)
		return 0;

	// CALCULATE BITMAP SIZE 
	// Align pitch to DWORD  

	bitmap_pitch = (sizl.cx << (ppdev->cPelSize)) + 3 & 0xFFFFFFFC;
	bitmap_size  = bitmap_pitch * sizl.cy;

	if (heap_index == ddraw_index || (bitmap_size + heap_offset) > ddraw_offset)
	{
		// OOPS, NO MORE ROOM 
		// We might be able to free up some room by repacking the data. 
        //
		if (invalid_count)
		{
			DISPDBG((ppdev, 3000, "OOPS, Repacking cache...\n"));

			RepackCache();

			// STILL NO ROOM? 
            //
			if ((bitmap_size + heap_offset) > ddraw_offset)
			{
				DISPDBG((ppdev, 3000, "Ran out of room in bitmap cache!!!!\n"));
				return 0;
			}
		}
		else
		{ 
			DISPDBG((ppdev, 3000, "Ran out of room in bitmap cache!!!!\n"));
			return 0;
		}
	}

	// USE A CODE FOR THE DHSURF 
	// We are trying a little trick here.  Rather than allocating memory 
	// for each surface and sending a pointer, we are using an unsigned  
	// long value.  This value is an index into an array.  To keep the   
	// pointer non-zero, we start the indexes at 2.   Index 1 is         
	// reserved as the dhsurf value for the primary surface.             
	//
    index = dhsurf_start;
    while (!(dhsurf_array[index] & CACHE_FLAG_INVALID))
        index++;

    dhsurf_start = index + 1;

	device_bitmap = EngCreateDeviceBitmap((DHSURF)(index | mode_value), sizl, iFormat);

	if (device_bitmap == NULL)
	{
		DISPDBG ((ppdev, 3000, "Could not create device bitmap!!\n"));
		return 0;
	}

	// ASSOCIATE THE BITMAP
	//
    if (!EngAssociateSurface (
        (HSURF)device_bitmap,
        ppdev->hdevEng,
        ppdev->flHooks))
    {
        EngDeleteSurface ((HSURF)device_bitmap);
		DISPDBG((ppdev, 3000, "EngAssociateSurface Failed!!!!\n"));
		return 0;
    }

	DISPDBG((ppdev, 30, "EngAssociateSurface Succeeded!!!!\n"));

	// ADD DATA TO HEAP ARRAY 
	//
	bitmap_heap[heap_index].dhsurf             = index;
	bitmap_heap[heap_index].flags_and_pitch    = bitmap_pitch;
	bitmap_heap[heap_index].width_and_height   = ((sizl.cx << 16) | sizl.cy);
	bitmap_heap[heap_index].bitmap_size        = bitmap_size;
    bitmap_heap[heap_index].lockedSurf         = NULL;
    bitmap_heap[heap_index].heap_offset        = heap_offset;

    // POINT DHSURF ARRAY TO HEAP ARRAY

    dhsurf_array[index] = heap_index;

	heap_index++;
	heap_offset += bitmap_size;

	return device_bitmap;
}

/////////////////////////////////////////////////////////////////////////
// DrvDeleteDeviceBitmap
//
// Routine responsible for deleting device-managed surfaces.
//
VOID DrvDeleteDeviceBitmap (DHSURF dhsurf)
{
	PDEV *ppdev = NULL;
	unsigned long config, misc;
	unsigned long unlock, gcfg;
    unsigned long index;
	unsigned long temp;

	// DELETE THE SURFACE 
	// We have two scenarios.  In scenario 1, the bitmap that is being  
	// deleted is at the end of the cache. In the second scenario, the  
	// bitmap is in the middle.  In an attempt to optimize performance, 
	// we will simply tag the data as invalid.  Later, when the cache   
	// fills up, we will make one call to repack all data.  The idea is 
	// that we will move less data that way, as each valid cache entry  
	// is only moved once every million times or so, and each bitmap    
	// is only moved once.  Also, it is rather tedious to modify the    
	// actual bitmap pointers such that GDI is drawing to the right     
	// space.  This prevents us from resetting each pointer each time   
	// a bitmap is deleted from the middle of the cache.                

	// CHECK IF BITMAP BELONGS TO A PREVIOUS MODE 
    //
	if (((ULONG)dhsurf >> 16) != mode_number)
	{
		DISPDBG ((ppdev, 3000, "Deleting Bitmap from previous mode.\n"));
		
        index = dhsurf_array[(unsigned short)dhsurf];

        if (index & CACHE_FLAG_SYSTEM)
        {
            index &= 0xFFFF;
            if (system_heap[index].lockedSurf)
            {
                EngUnlockSurface (system_heap[index].lockedSurf);
                EngDeleteSurface ((HSURF)system_heap[index].punt_surface);
            }
            EngFreeMem ((PVOID)system_heap[index].heap_offset);
        }
        else
        {
            index &= 0xFFFF;
            if (bitmap_heap[index].lockedSurf)
            {
                EngUnlockSurface (bitmap_heap[index].lockedSurf);
                EngDeleteSurface ((HSURF)bitmap_heap[index].punt_surface);
            }
        }
        
        old_mode_count--;

		if (old_mode_count == 0)
		{
			DISPDBG ((ppdev, 3000, "All bitmaps deleted.\n"));

			// Enable the display and clear the screen

            ACQUIRE_PRIMITIVE_SEMAPHORE;

			GU2_WAIT_PENDING;
			WRITE_GP32 (MGP_RASTER_MODE, gu2_bpp);
			WRITE_GP32 (MGP_WID_HEIGHT, ((screen_width << 16) | (screen_height & 0xFFFF)));
			WRITE_GP32 (MGP_DST_OFFSET, 0);
			WRITE_GP32 (MGP_STRIDE, mode_pitch);
			WRITE_GP32 (MGP_BLT_MODE, 0);

            RELEASE_PRIMITIVE_SEMAPHORE;

			config = READ_VID32(RCDF_DISPLAY_CONFIG);
			misc   = READ_VID32(RCDF_VID_MISC);
	
			WRITE_VID32(RCDF_DISPLAY_CONFIG, config | RCDF_DCFG_DIS_EN   | RCDF_DCFG_HSYNC_EN
												    | RCDF_DCFG_VSYNC_EN | RCDF_DCFG_DAC_BL_EN);
			WRITE_VID32(RCDF_VID_MISC, misc & ~RCDF_DAC_POWER_DOWN & ~RCDF_ANALOG_POWER_DOWN);

			// Enable compression

			if (compression_enabled)
			{
				// Clear the dirty/valid bits

				temp = READ_REG32 (MDC_DV_CTL);
				WRITE_REG32 (MDC_DV_CTL, temp | 0x00000001);

				// Turn on compression control bits

				unlock = READ_REG32 (MDC_UNLOCK);
				gcfg   = READ_REG32 (MDC_GENERAL_CFG);
				gcfg  |= MDC_GCFG_CMPE | MDC_GCFG_DECE;
				WRITE_REG32 (MDC_UNLOCK, MDC_UNLOCK_VALUE);
				WRITE_REG32 (MDC_GENERAL_CFG, gcfg);
				WRITE_REG32 (MDC_UNLOCK, unlock);
			}
		}

		else if (old_mode_count < 0)
			DISPDBG ((ppdev, 3000, "Too many bitmaps deleted.\n"));

		return;
	}	

    index = (ULONG)dhsurf & 0xFFFF;

    if (index < dhsurf_start)
        dhsurf_start = index;

    index = dhsurf_array[index];
    dhsurf_array[(unsigned short)dhsurf] = CACHE_FLAG_INVALID;

    if ((index & 0xFFFF) >= HEAP_MAX)
    {
        DISPDBG ((ppdev, 3000, "Invalid Index in dhsurf array!\n"));
        return;
    }

    if (index & CACHE_FLAG_SYSTEM)
    {
        index &= 0xFFFF;
        if (system_heap[index].lockedSurf)
        {
            EngUnlockSurface (system_heap[index].lockedSurf);
            EngDeleteSurface ((HSURF)system_heap[index].punt_surface);
        }
        if (index == (system_index - 1))
            system_index--;
        else
        {
            system_heap[index].flags_and_pitch |= CACHE_FLAG_INVALID;
            system_invalid++;
        }
        EngFreeMem ((PVOID)system_heap[index].heap_offset);
    }
    else
    {
        index &= 0xFFFF;
        if (bitmap_heap[index].lockedSurf)
        {
            EngUnlockSurface (bitmap_heap[index].lockedSurf);
            EngDeleteSurface ((HSURF)bitmap_heap[index].punt_surface);
        }
        if (index == (heap_index - 1))
        {
            heap_offset -= bitmap_heap[index].bitmap_size;
		    heap_index--;
        }
        else 
        {	
		    bitmap_heap[index].flags_and_pitch |= CACHE_FLAG_INVALID;
		    invalid_count++;
	    }
    }
}

/////////////////////////////////////////////////////////////////////////
// AddLockedSurfForSystemBitmap
//
// Local routine to create a locked surface around a device bitmap to allow 
// punting a device bitmap to GDI.
//
int AddLockedSurfForSystemBitmap (PDEV *ppdev, int index)
{
    HBITMAP hbmp;
    SURFOBJ *pso;
    SIZEL sizl;

    sizl.cx = system_heap[index].width_and_height >> 16;
    sizl.cy = system_heap[index].width_and_height & 0xFFFF;
    hbmp = EngCreateBitmap (
        sizl,
        system_heap[index].flags_and_pitch & 0xFFFF,  
        ppdev->iBitmapFormat,
        BMF_TOPDOWN,
        (PVOID)(system_heap[index].heap_offset));

    if (!hbmp)
    {
        DISPDBG ((ppdev, 3000, "Unable to create bitmap\n"));
        return 0;
    }

    pso = EngLockSurface ((HSURF)hbmp);

    if (!pso)
    {
        DISPDBG ((ppdev, 3000, "Unable to lock surface\n"));
        EngDeleteSurface ((HSURF)hbmp);
        return 0;
    }

    system_heap[index].lockedSurf = pso;
    system_heap[index].punt_surface = hbmp;

    return 1;
}

/////////////////////////////////////////////////////////////////////////
// AddLockedSurf
//
// Local routine to create a locked surface around a device bitmap to allow 
// punting a device bitmap to GDI.
//
int AddLockedSurf (PDEV *ppdev, int index)
{
    HBITMAP hbmp;
    SURFOBJ *pso;
    SIZEL sizl;

    sizl.cx = bitmap_heap[index].width_and_height >> 16;
    sizl.cy = bitmap_heap[index].width_and_height & 0xFFFF;
    hbmp = EngCreateBitmap (
        sizl,
        bitmap_heap[index].flags_and_pitch & 0xFFFF,  
        ppdev->iBitmapFormat,
        BMF_TOPDOWN,
        gfx_virt_fbptr + bitmap_heap[index].heap_offset);

    if (!hbmp)
    {
        DISPDBG ((ppdev, 3000, "Unable to create bitmap\n"));
        return 0;
    }

    pso = EngLockSurface ((HSURF)hbmp);

    if (!pso)
    {
        DISPDBG ((ppdev, 3000, "Unable to lock surface\n"));
        EngDeleteSurface ((HSURF)hbmp);
        return 0;
    }

    bitmap_heap[index].lockedSurf = pso;
    bitmap_heap[index].punt_surface = hbmp;

    return 1;
}

/////////////////////////////////////////////////////////////////////////
// RepackCache
//
// Local routine to pack fragmented bitmaps into the beginning of offscreen memory.
//
void RepackCache (void)
{
	unsigned long i, j;
	unsigned long srcoffset, dstoffset;
    unsigned long stride;
    PDEV *ppdev = NULL;

    int puntCount = 0;

	// WRITE THE GP REGISTERS THAT DON'T CHANGE 
	// We are doing a screen to screen copy for each bitmap.        
	// This differs from the Win98 strategy as the Win98 driver     
	// repacked the cache each time a bitmap was deleted from       
	// the middle.  This can be inefficient when deleting multiple  
	// bitmaps from the center of the cache.  The new way uses more 
	// BLTs but copies less data.                                   
    //
    ACQUIRE_PRIMITIVE_SEMAPHORE;
    
    GU2_WAIT_PENDING;
	WRITE_GP32 (MGP_RASTER_MODE, (gu2_bpp | 0xCC));

	// START AT BEGINNING OF HEAP 
    //
	srcoffset = dstoffset = heap_start;

	j = 2;
	for (i = 2; i < heap_index; i++)
	{
		// CHECK IF BITMAP IS INVALID 
		// If the bitmap has been deleted, we ignore it and increment our source  
		// pointer.  If the bitmap is valid and the dest and source are not equal 
		// we will copy the bitmap cache entry and the corresponding data to the  
		// destination.  If the dest and source pointers are equivalent, we have  
		// not yet identified an invalid bitmap and we keep going.                
        //
		if (bitmap_heap[i].flags_and_pitch & CACHE_FLAG_INVALID)
			srcoffset += bitmap_heap[i].bitmap_size;
		
		else if (srcoffset == dstoffset)
		{
			srcoffset += bitmap_heap[i].bitmap_size;
			dstoffset += bitmap_heap[i].bitmap_size;
			j++;
		}
		else
		{
			// BITMAP COPY! 
			// We do the copy in three steps.      
			// 1. Copy the bitmap data.            
			// 2. Copy the bitmap cache data.
            // 3. Adjust the dhsurf array.
            //
            stride = (bitmap_heap[i].flags_and_pitch << 16) | 
				     (bitmap_heap[i].flags_and_pitch & 0xFFFF);

			// Kick off the BLT

			GU2_WAIT_PENDING;
			WRITE_GP32 (MGP_SRC_OFFSET, srcoffset);
			WRITE_GP32 (MGP_DST_OFFSET, dstoffset);
			WRITE_GP32 (MGP_WID_HEIGHT, bitmap_heap[i].width_and_height);
			WRITE_GP32 (MGP_STRIDE, stride);
			WRITE_GP32 (MGP_BLT_MODE, MGP_BM_SRC_FB);
			
			// COPY HEAP DATA 
            //
			bitmap_heap[j] = bitmap_heap[i];
            bitmap_heap[j].heap_offset = dstoffset;

			// ADJUST DHSURF POINTER

            dhsurf_array[bitmap_heap[j].dhsurf] = j;

            // DELETE PUNT SURFACE
            // The punt surface was created from a pointer to the old heap offset.  After
            // moving the data, that pointer is now invalid.  We are making the calculated
            // assumption that punts are rare, in which case, there will be very few punt 
            // surfaces.  In that case, it is optimal to simply delete the old punt surface
            // and hope that no more punts occur to this surface.
            //
            if (bitmap_heap[j].lockedSurf)
            {
                EngUnlockSurface (bitmap_heap[j].lockedSurf);
                EngDeleteSurface ((HSURF)bitmap_heap[j].punt_surface);
                bitmap_heap[j].lockedSurf = NULL;

                puntCount++;
            }
			
			srcoffset += bitmap_heap[j].bitmap_size;
			dstoffset += bitmap_heap[j].bitmap_size;
			j++;
		}	
	}

    RELEASE_PRIMITIVE_SEMAPHORE;

	// SET THE NEW HEAP PARAMETERS                               
	// J is the new heap index.  Dstoffset is the new heap size.
    //
	heap_index  = j;
	heap_offset = dstoffset;

	invalid_count = 0;

    DISPDBG ((ppdev, 3000, "%i punt surfaces deleted\n", puntCount));
}

/////////////////////////////////////////////////////////////////////////
// RepackSystemMemoryHeap
//
// Local routine to repack the array holding system memory bitmaps.
//
void RepackSystemMemoryHeap (void)
{
    unsigned int i, j;

    j = 0;
    for (i = 0; i < system_index; i++)
    {
        if (!(system_heap[i].flags_and_pitch & CACHE_FLAG_INVALID))
        {
            if (i != j)
            {
                // Copy the data.
                //
                system_heap[j] = system_heap[i];

                // Adjust the dhsurf array.
                //
                dhsurf_array[system_heap[j].dhsurf] = j | CACHE_FLAG_SYSTEM;
            }

            j++;
        }
    }

    system_invalid = 0;
    system_index   = j;
}

/////////////////////////////////////////////////////////////////////////
// MoveBitmapToSystemMemory
//
// Local routine to move a device-managed bitmap into system memory to free up 
// heap space.  This routine is called on DirectDraw surface creation and is
// not viewed to be time-critical.
//
int MoveBitmapToSystemMemory (int index)
{
	PDEV *ppdev = NULL;
	unsigned long fboffset;
	int i;
	VOID *pvScan0;

	// ALLOCATE THE BUFFER 
	// Do NOT zero-initialize the buffer.  We're about to 
	// write to the entire thing.                         
    //
	pvScan0 = EngAllocMem(0, bitmap_heap[index].bitmap_size, 0x20786745);

	if (pvScan0 == NULL)
	{
		DISPDBG ((ppdev, 3000, "Unable to allocate memory for system bitmap\n"));
		return 0;
	}

	// EXTRACT THE BITMAP OFFSET AND COPY THE DATA
	//
    fboffset = bitmap_heap[index].heap_offset;
	
	memcpy (pvScan0, (void *)(gfx_virt_fbptr + fboffset), bitmap_heap[index].bitmap_size);

    // UPDATE THE DHSURF ARRAY
    // This is the tricky part.  We do not want to change the way GDI views the surface.
    // Instead, we want to update the dhsurf array to point to the array of system memory
    // surfaces.
    //
    if (system_index == HEAP_MAX)
    {
        if (system_invalid)
            RepackSystemMemoryHeap();
        else
        {
            EngFreeMem (pvScan0);
            DISPDBG ((ppdev, 10000, "VERY VERY BAD!!! No room in system memory bitmap cache???\n"));
            return 0;
        }
    }

    system_heap[system_index].heap_offset      = (ULONG)pvScan0;
    system_heap[system_index].flags_and_pitch  = bitmap_heap[index].flags_and_pitch & 0xFFFF;
    system_heap[system_index].width_and_height = bitmap_heap[index].width_and_height;
    system_heap[system_index].lockedSurf       = NULL;
    system_heap[system_index].dhsurf           = bitmap_heap[index].dhsurf;
    system_heap[system_index].bitmap_size      = bitmap_heap[index].bitmap_size;

    dhsurf_array[system_heap[system_index].dhsurf] = system_index | CACHE_FLAG_SYSTEM;

    system_index++;

    return 1;
}

/////////////////////////////////////////////////////////////////////////
// EmptyBitmapHeap
//
// This routine resets the bitmap heap.  The mode index is incremented such that all bitmaps 
// created after this point will be marked as using the current mode.  
//
void EmptyBitmapHeap (PDEV *ppdev, unsigned long new_heap_start)
{
	unsigned long i;
	
	// SAVE THE REMAINING BITMAP COUNT 
	// We will fail all calls to DrvCreateDeviceBitmap until all bitmaps 
	// have been deleted.  
    //
	old_mode_count = (heap_index - invalid_count - 2) + (system_index - system_invalid);

	// SET THE NEW HEAP VALUES                  
	// Assume that all bitmaps are now invalid. 
    //	
	heap_index     = 2;
	heap_offset    = new_heap_start;
	invalid_count  = 0;
    system_index   = 0;
    system_invalid = 0;

	if (new_heap_start >= heap_end)
		heap_start = 0;
	else
		heap_start = new_heap_start;
	
    // MARK ALL DHSURF ENTRIES AS INVALID

    for (i = 0; i < HEAP_MAX; i++)
        dhsurf_array[i] |= CACHE_FLAG_INVALID;

    dhsurf_start = 2;

	// INCREMENT THE MODE NUMBER 
	// Windows is rather illogical at this point.  All bitmap handles are 
	// invalid (the bitmaps have been deleted) but its heap manager will  
	// not call DrvDeleteDeviceBitmap until after the mode has been set.  
	// This would cause problems if we attempted to repack before all     
	// invalid bitmaps had been deleted.                                  
    //
	mode_number = (mode_number + 1) & 3;
	mode_value  =  mode_number << 16;
}

/////////////////////////////////////////////////////////////////////////
// SetPrimaryHeapEntry 
//
// Local routine to fill in the first entry of the bitmap heap with information
// about the primary surface.
//
void SetPrimaryHeapEntry (PDEV *ppdev)
{
    // PRESET THE FIRST HEAP ENTRY 

    dhsurf_array[1]                   = 1;
    bitmap_heap[1].punt_surface       = ppdev->GXBitmap;
    bitmap_heap[1].lockedSurf         = ppdev->pLockedSurf;
    bitmap_heap[1].dhsurf             = 1;
    bitmap_heap[1].flags_and_pitch    = ppdev->lDelta;
    bitmap_heap[1].width_and_height   = (ppdev->cxScreen << 16) | ppdev->cyScreen;
    bitmap_heap[1].bitmap_size        = ppdev->lDelta * ppdev->cyScreen;
    bitmap_heap[1].heap_offset        = 0;
}

/////////////////////////////////////////////////////////////////////////
// AddDDrawBitmap
//
// Local routine to allocate space in the bitmap heap to a DirectDraw 
// bitmap.
//
int AddDDrawBitmap (PDEV *ppdev, unsigned long bitmap_size, unsigned long pitch)
{
	unsigned long width_and_height;

	// CHECK IF THERE IS NO HEAP 
    //
	if (!heap_start)
		return 0;

	// CHECK IF BITMAP HEAP ARRAY IS FULL 
    //
	if (ddraw_index < heap_index)
	{
		// CHECK IF HEAP IS FULL 
        //
		if (heap_index == 2)
		{
			DISPDBG ((ppdev, 3000, "Not enough entries for DDraw bitmap.\n"));
			return 0;
		}

		// TRY TO REPACK THE GDI BITMAPS 
        //
		if (invalid_count)
			RepackCache();
		
		// REMOVE A BITMAP TO SYSTEM MEMORY 
		//
		if (ddraw_index < heap_index)
		{
			if (MoveBitmapToSystemMemory (heap_index - 1))
			{
				heap_index--;

                if (bitmap_heap[heap_index].lockedSurf)
                {
                    EngUnlockSurface (bitmap_heap[heap_index].lockedSurf);
                    EngDeleteSurface ((HSURF)bitmap_heap[heap_index].punt_surface);
                }

				heap_offset -= bitmap_heap[heap_index].bitmap_size;
			}
			else
			{
				DISPDBG ((ppdev, 3000, "Unable to move bitmap to system memory.\n"));
				return 0;
			}
		}
	}

	// CHECK IF ENOUGH ROOM 
    //
	if ((ddraw_offset - bitmap_size) < heap_offset)
	{
		// CHECK IF THERE WILL EVER BE ROOM 
        //
		if ((ddraw_offset - bitmap_size) < heap_start)
			return 0;

		// TRY TO REPACK THE GDI HEAP 
        //
		if (invalid_count)
			RepackCache();

		// MOVE BITMAPS OUT 
        //
		while ((ddraw_offset - bitmap_size) < heap_offset)
		{
			if (!MoveBitmapToSystemMemory (heap_index - 1))
			{
				DISPDBG ((ppdev, 3000, "Unable to move bitmap out!!!!!\n"));
				return 0;
			}

			heap_index--;

            DISPDBG ((ppdev, 3000, "Deleting dhsurf %i\n", bitmap_heap[heap_index].dhsurf));

            if (bitmap_heap[heap_index].lockedSurf)
            {
                EngUnlockSurface (bitmap_heap[heap_index].lockedSurf);
                EngDeleteSurface ((HSURF)bitmap_heap[heap_index].punt_surface);
            }

			heap_offset -= bitmap_heap[heap_index].bitmap_size;
		}
	}
	
	width_and_height  = (pitch >> ppdev->cPelSize) << 16;
	width_and_height |=  bitmap_size / pitch;
	bitmap_heap[ddraw_index].bitmap_size      = bitmap_size;
	bitmap_heap[ddraw_index].flags_and_pitch  = pitch;
	bitmap_heap[ddraw_index].width_and_height = width_and_height;
		
	ddraw_index--;
	ddraw_offset -= bitmap_size;

	// UPDATE BITMAP COUNT 
	// As we have no way of updating locked surface pointers, we 
	// will wait until all surfaces have been deleted and reset  
	// the heap.                                                 
    //
	ddraw_count++;

	return 1;
}

/////////////////////////////////////////////////////////////////////////
// ResetDDrawHeap
//
// Local routine to reset the contents of the DirectDraw heap.  This routine
// is called when a DirectDraw surface is deleted and is not viewed to be 
// time-critical.
//
void ResetDDrawHeap (void)
{
    unsigned long i;
    PDEV *ppdev = NULL;

	ddraw_count   = 0;
	ddraw_invalid = 0;
	ddraw_offset  = heap_end;
	ddraw_index   = ddraw_max;

    // REINSTATE ANY EJECTED GDI BITMAPS
    // We purposely left ejected surfaces opaque such that we could delete
    // the memory when the surfaces were no longer needed.  A positive side
    // effect of this is that we can reinstate the bitmaps as soon as the
    // DirectDraw heap is no longer being used.
    // 
    if (system_invalid)
        RepackSystemMemoryHeap();
    if (invalid_count)
        RepackCache();

    for (i = 0; i < system_index; i++)
    {
        // Check if there will be room for the new bitmap
        //
        if ((system_heap[i].bitmap_size + heap_offset) > ddraw_offset || heap_index == ddraw_index)
            continue;

        DISPDBG ((ppdev, 3000, "Reinstating a GDI bitmap\n"));

        // Add a new VRAM heap entry
        //
        bitmap_heap[heap_index] = system_heap[i];
        bitmap_heap[heap_index].lockedSurf = NULL;
        bitmap_heap[heap_index].heap_offset = heap_offset;

        // Copy the data back to the frame buffer
        //
        memcpy ((PVOID)(gfx_virt_fbptr + heap_offset), (PVOID)system_heap[i].heap_offset, 
            system_heap[i].bitmap_size);

        // Update the dhsurf array
        //
        dhsurf_array[system_heap[i].dhsurf] = heap_index;

        // Delete any punt surfaces
        //
        if (system_heap[i].lockedSurf)
        {
            EngUnlockSurface (system_heap[i].lockedSurf);
            EngDeleteSurface ((HSURF)system_heap[i].punt_surface);
        }
        EngFreeMem ((PVOID)system_heap[i].heap_offset);

        // Mark the system heap entry as invalid and update the main heap.
        //
        system_heap[i].flags_and_pitch |= CACHE_FLAG_INVALID;
        system_invalid++;
        heap_index++;
        heap_offset += system_heap[i].bitmap_size;
    }
}
