 /*
  * <LIC_AMD_STD>
  * Copyright (C) <years> Advanced Micro Devices, Inc.  All Rights Reserved.
  * </LIC_AMD_STD>
  * 
  * <CTL_AMD_STD>
  * </CTL_AMD_STD>
  * 
  * <DOC_AMD_STD>
  * Driver entry and display mode routines.
  * </DOC_AMD_STD>
  * 
  */

#include "precomp.h"
#include "gfx_regs.h"
#include "gfx_defs.h"

/*-------------------------------------------------------------------------
 * GDIINFO ggdiDefault
 *
 * This structure contains the default GDIINFO fields that are passed back to GDI
 * during DrvEnablePDEV.
 *
 * NOTE: This structure defaults to values for an 8bpp palette device.
 *		 Some fields are overwritten for different color depths.
 -------------------------------------------------------------------------*/

GDIINFO ggdiDefault = {
	GDI_DRIVER_VERSION,		/* ulVersion (3.51, 4.0, etc.     */
	DT_RASDISPLAY,			/* ulTechnology                   */
	0,						/* ulHorzSize (filled in later)   */
	0,						/* ulVertSize (filled in later)   */
	0,						/* ulHorzRes (filled in later)    */
	0,						/* ulVertRes (filled in later)    */
	0,						/* cBitsPixel (filled in later)   */
	0,						/* cPlanes (filled in later)      */
	20, 					/* ulNumColors (palette managed)  */
	0,						/* flRaster (DDI reserved field)  */

	0,						/* ulLogPixelsX (filled in later) */
	0,						/* ulLogPixelsY (filled in later) */

	/* flTextCaps -- If we had wanted console windows to scroll by repainting */
	/* the entire window, instead of doing a screen-to-screen blt, we         */
	/* would have set TC_SCROLLBLT (yes, the flag is bass-ackwards).          */
	
	TC_RA_ABLE,

	0,						/* ulDACRed   (filled in later)             */
	0,						/* ulDACGreen (filled in later)             */
	0,						/* ulDACBlue  (filled in later)             */

	0x0024, 				/* ulAspectX                                */
	0x0024, 				/* ulAspectY                                */
	0x0033, 				/* ulAspectXY (one-to-one aspect ratio)     */

	1,						/* xStyleStep                               */
	1,						/* yStyleStep                               */
	3,						/* denStyleStep -- Styles have a one-to-one */
	                        /*                 aspect ratio, and every  */
							/*                 'dot' is 3 pixels long.  */

	{ 0, 0 },				/* ptlPhysOffset                            */
	{ 0, 0 },				/* szlPhysSize                              */

	256,					/* ulNumPalReg                              */

	/* These fields are for halftone initialization.  The actual values are */
	/* a bit magic, but seem to work well on our display.                   */
	 
	{						/* ciDevice            */
	   { 6700, 3300, 0 },	/*		Red            */
	   { 2100, 7100, 0 },	/*		Green          */
	   { 1400,	800, 0 },	/*		Blue           */
	   { 1750, 3950, 0 },	/*		Cyan           */
	   { 4050, 2050, 0 },	/*		Magenta        */
	   { 4400, 5200, 0 },	/*		Yellow         */
	   { 3127, 3290, 0 },	/*		AlignmentWhite */
	   20000,				/*		RedGamma       */
	   20000,				/*		GreenGamma     */
	   20000,				/*		BlueGamma      */
	   0, 0, 0, 0, 0, 0 	/*		No dye correction for raster displays */
	},

	0,						 /* ulDevicePelsDPI (for printers only) */
	PRIMARY_ORDER_CBA,		 /* ulPrimaryOrder                      */
	HT_PATSIZE_4x4_M,		 /* ulHTPatternSize                     */
	HT_FORMAT_8BPP, 		 /* ulHTOutputFormat                    */
	HT_FLAG_ADDITIVE_PRIMS,  /* flHTFlags                           */
	0,						 /* ulVRefresh                          */
	0,						 /* ulBltAlignment                      */
	0,						 /* ulPanningHorzRes                    */
	0,						 /* ulPanningVertRes                    */
};

/* SYSTEM FONTS */

#define SYSTM_LOGFONT {16,7,0,0,700,0,0,0,ANSI_CHARSET,OUT_DEFAULT_PRECIS,\
					   CLIP_DEFAULT_PRECIS,DEFAULT_QUALITY,\
					   VARIABLE_PITCH | FF_DONTCARE,L"System"}
#define HELVE_LOGFONT {12,9,0,0,400,0,0,0,ANSI_CHARSET,OUT_DEFAULT_PRECIS,\
					   CLIP_STROKE_PRECIS,PROOF_QUALITY,\
					   VARIABLE_PITCH | FF_DONTCARE,L"MS Sans Serif"}
#define COURI_LOGFONT {12,9,0,0,400,0,0,0,ANSI_CHARSET,OUT_DEFAULT_PRECIS,\
					   CLIP_STROKE_PRECIS,PROOF_QUALITY,\
					   FIXED_PITCH | FF_DONTCARE, L"Courier"}

/*-------------------------------------------------------------------------
 * DEVINFO gdevinfoDefault
 *
 * This contains the default DEVINFO fields that are passed back to GDI
 * during DrvEnablePDEV.
 *
 * NOTE: This structure defaults to values for an 8bpp palette device.
 *		Some fields are overwritten for different color depths.
 -------------------------------------------------------------------------*/

DEVINFO gdevinfoDefault = {
	(GCAPS_OPAQUERECT		|
	 GCAPS_MONO_DITHER      |
	 GCAPS_GRAY16),
	SYSTM_LOGFONT,								/* lfDefaultFont */
	HELVE_LOGFONT,								/* lfAnsiVarFont */
	COURI_LOGFONT,								/* lfAnsiFixFont */
	0,											/* cFonts */
	BMF_8BPP,									/* iDitherFormat */
	8,											/* cxDither */
	8,											/* cyDither */
	0											/* hpalDefault (filled in later) */
};

/*-------------------------------------------------------------------------
 * DFVFN gadrvfn[]
 *
 * Build the driver function table gadrvfn with function index/address
 * pairs.  This table tells GDI which DDI calls we support, and their
 * location (GDI does an indirect call through this table to call us).
 -------------------------------------------------------------------------*/

DRVFN gadrvfn[] = {
	{	INDEX_DrvEnablePDEV,			(PFN) DrvEnablePDEV 		    },
	{	INDEX_DrvCompletePDEV,			(PFN) DrvCompletePDEV		    },
	{	INDEX_DrvDisablePDEV,			(PFN) DrvDisablePDEV		    },
	{	INDEX_DrvEnableSurface, 		(PFN) DrvEnableSurface		    },
	{	INDEX_DrvDisableSurface,		(PFN) DrvDisableSurface 	    },
	{	INDEX_DrvCreateDeviceBitmap,	(PFN) DrvCreateDeviceBitmap     },
	{	INDEX_DrvDeleteDeviceBitmap,	(PFN) DrvDeleteDeviceBitmap     },
	{	INDEX_DrvAssertMode,			(PFN) DrvAssertMode 		    },
	{	INDEX_DrvMovePointer,			(PFN) DrvMovePointer		    },
	{	INDEX_DrvSetPointerShape,		(PFN) DrvSetPointerShape	    },
	{	INDEX_DrvSetPalette,			(PFN) DrvSetPalette 		    },
	{	INDEX_DrvCopyBits,				(PFN) DrvCopyBits			    },
	{	INDEX_DrvStrokePath,			(PFN) DrvStrokePath			    },
	{	INDEX_DrvTextOut,				(PFN) DrvTextOut			    },
	{	INDEX_DrvBitBlt,				(PFN) DrvBitBlt 			    },
	{	INDEX_DrvGetModes,				(PFN) DrvGetModes			    },
	{	INDEX_DrvRealizeBrush,			(PFN) DrvRealizeBrush		    },
	{	INDEX_DrvEscape,				(PFN) DrvEscape 			    },
    {   INDEX_DrvGetDirectDrawInfo,     (PFN) DrvGetDirectDrawInfo      },
    {   INDEX_DrvEnableDirectDraw,      (PFN) DrvEnableDirectDraw       },
    {   INDEX_DrvDisableDirectDraw,     (PFN) DrvDisableDirectDraw      },
	{   INDEX_DrvAlphaBlend,            (PFN) DrvAlphaBlend             },
	{   INDEX_DrvTransparentBlt,        (PFN) DrvTransparentBlt         },
    {   INDEX_DrvDisableDriver,         (PFN) DrvDisableDriver          },
    {   INDEX_DrvLineTo,                (PFN) DrvLineTo                 },
    {   INDEX_DrvIcmSetDeviceGammaRamp, (PFN) DrvIcmSetDeviceGammaRamp  },
};

ULONG gcdrvfn = sizeof(gadrvfn) / sizeof(DRVFN);

/* MEMORY ALLOCATION TAG                                                   */
/* This is a unique tag that is prepended to any area of memory allocation */
/* to uniquely identify the block as belonging to us.                      */

ULONG gAllocTag = 0x20786744;	/* 'Dgx ' */

static DWORD getAvailableModes(PDEV *ppdev, HANDLE hDriver,
		PVIDEO_MODE_INFORMATION* modeInformation, DWORD *cbModeSize);
	
BOOL bInitPointer(PDEV *ppdev, DEVINFO * pdevinfo);	

/* DURANGO POINTERS */

unsigned char *gfx_virt_gpptr;
unsigned char *gfx_virt_regptr;
unsigned char *gfx_virt_vidptr;
unsigned char *gfx_virt_fbptr;
unsigned long  gfx_fb_size;
unsigned long  gu2_bpp;

/* INITIALIZATION FLAG                       */
/* We want to initialize the heap only ONCE! */

unsigned char mode_shift;
int previous_width  = 0;
int previous_height = 0;
int previous_bpp    = 0;

void EmptyBitmapHeap (PDEV *ppdev, unsigned long new_heap_start);
void ResetDDrawHeap (void);

/*-------------------------------------------------------------------------
 * DrvEnableDriver
 *
 * Enables the driver by retrieving the drivers function table and version.
 -------------------------------------------------------------------------*/

BOOL DrvEnableDriver(
	ULONG		   iEngineVersion,
	ULONG		   cj,
	DRVENABLEDATA* pded)
{
	PDEV *ppdev = NULL;

	DISPDBG ((ppdev, 1, "DrvEnableDriver Entry\n"));

	/* FILL IN ENGINE VERSION                                                  */
	/* Engine Version is passed down so future drivers can support previous    */
	/* engine versions.  A next generation driver can support both the old     */
	/* and new engine conventions if told what version of engine it is         */
	/* working with.  For the first version the driver does nothing with it.   */
	 
	if(cj >= sizeof(DRVENABLEDATA))
		pded->pdrvfn = gadrvfn;

	if(cj >= (sizeof(ULONG) * 2))
		pded->c = gcdrvfn;

	/* FILL IN DDI VERSION                                                       */
	/* For Win2K and XP, we set the engine version to NT4.0  This can be changed */
	/* but must be tested.                                                       */ 
	 
#if _WIN32_WINNT >= 0x0500
	if(cj >= sizeof(ULONG))
		pded->iDriverVersion = DDI_DRIVER_VERSION_NT4;
#else
	if(cj >= sizeof(ULONG))
		pded->iDriverVersion = DDI_DRIVER_VERSION;
#endif 

	return TRUE;
}

/*-------------------------------------------------------------------------
 * VOID DrvDisableDriver
 *
 * Tells the driver it is being disabled. Release any resources allocated in
 * DrvEnableDriver.
 *
 -------------------------------------------------------------------------*/

VOID DrvDisableDriver (VOID)
{
	PDEV *ppdev = NULL;		/* To satisfy debug code */

	DISPDBG ((ppdev, 1, "DrvDisableDriver\n"));
	return;
}

/*-------------------------------------------------------------------------
 * DHPDEV DrvEnablePDEV
 *
 * Initializes a bunch of fields for GDI, based on the mode we've been asked
 * to do.  This is the first thing called after DrvEnableDriver, when GDI
 * wants to get some information about us.
 *
 * (This function mostly returns back information; DrvEnableSurface is used
 * for initializing the hardware and driver components.)
 -------------------------------------------------------------------------*/

DHPDEV DrvEnablePDEV(
	DEVMODEW*	pdm,			/* Contains data pertaining to requested mode      */
	PWSTR		pwszLogAddr,	/* Logical address                                 */
	ULONG		cPat,			/* Count of standard patterns                      */
	HSURF*		phsurfPatterns, /* Buffer for standard patterns                    */
	ULONG		cjCaps, 		/* Size of buffer for device caps 'pdevcaps'       */
	ULONG*		pdevcaps,		/* Buffer for device caps, also known as 'gdiinfo' */
	ULONG		cjDevInfo,		/* Number of bytes in device info 'pdi'            */
	DEVINFO*	pdi,			/* Device information                              */
	HDEV		hdev,			/* Hardware device                                 */
	PWSTR		pwszDeviceName, /* Device name                                     */
	HANDLE		hDriver)		/* Kernel driver handle                            */
{
	PDEV	*ppdev = NULL;
	DWORD ReturnedDataLength;

	DISPDBG ((ppdev, 1, "DrvEnablePDEV\n"));
	
	/* TEST SIZE OF GDIINFO STRUCTURE */
	/* Future versions of Windows should provide structures that are the same size */
	/* or greater than what we expect.                                             */

	if((cjCaps < sizeof(GDIINFO)) || (cjDevInfo < sizeof(DEVINFO))) 
	{
		DISPDBG((ppdev, 300, "DrvEnablePDEV - Buffer size too small\n"));
		return 0;
	}

	/* ALLOCATE A PHYSICAL DEVICE STRUCTURE */

	ppdev = (PDEV*) EngAllocMem (FL_ZERO_MEMORY, sizeof(PDEV), gAllocTag);

	/* CHECK LOW RESOURCE CASE */

	if(ppdev == NULL) 
	{
		DISPDBG((ppdev, 3000, "DrvEnablePDEV - Failed EngAllocMem\n"));
		return 0;
	}

#ifdef DBG
	/* INITIALIZE POINTER TO BEGINNING OF DEBUG LOG              */           
	/* End of string is indicated by one null, end of log is     */
	/* indicated by two nulls.                                   */

	ppdev->debugBegin = 0;
	ppdev->debugEnd   = 0;
	ppdev->debugMode  = DEBUG_MODE_NT;	/* Direct debugging to COM2 */
#endif

	ppdev->hDriver = hDriver;

	/* GET THE CURRENT SCREEN MODE INFORMATION */
	/* Also, set up device caps and devinfo.   */

	if (!bInitializeModeFields(ppdev, (GDIINFO*) pdevcaps, pdi, pdm)) 
	{
		DISPDBG((ppdev, 3000, "DrvEnablePDEV - Failed bInitializeModeFields\n"));
		goto ReturnFailure1;
	}

	/* INITIALIZE PALETTE INFORMATION */

	if (!bInitializePalette(ppdev, pdi)) 
	{
		DISPDBG((ppdev, 3000, "DrvEnablePDEV - Failed bInitializePalette\n"));
		goto ReturnFailure1;
	}

    /* INITIALIZE CURSOR INFORMATION */

	pdi->flGraphicsCaps  |= GCAPS_ASYNCMOVE;

	/* RETRIEVE DURANGO FUNCTION POINTERS */
	/* If Durango becomes included in the display driver, this will no */
	/* longer be necessary                                             */

#ifdef DURANGO
	if(EngDeviceIoControl(ppdev->hDriver,
						 IOCTL_GET_DURANGO_FUNCS,
						 &ppdev->myfuncs,  /* input buffer */
						 sizeof(DURANGO_FUNCS),
						 &ppdev->myfuncs,  /* output buffer */
						 sizeof(DURANGO_FUNCS),
						 &ReturnedDataLength))
	{
		DISPDBG((ppdev, 300, "bAssertModeHardware - Failed VIDEO_SET_CURRENT_MODE\n"));
		goto ReturnFailure1;
	}
#endif

	/* SET ANTIALIASED TEXT FLAG */

	if (ppdev->cBitsPerPel != 8 && ppdev->myfuncs.AntialiasedText)
		pdi->flGraphicsCaps |= GCAPS_GRAY16;
    else
        pdi->flGraphicsCaps &= ~GCAPS_GRAY16;


	/* SET PANNING FLAG */
	/* This will produce DrvMove Pointer calls for a software pointer */
	/* Only enable this flag for panned displays, as it will produce  */
	/* two pointer calls for each mouse move.                         */

	if (ppdev->cyScreen > *(ppdev->myfuncs.pPanelHeight) || 
		ppdev->cxScreen > *(ppdev->myfuncs.pPanelWidth))
		pdi->flGraphicsCaps |= GCAPS_PANNING;
	else
		pdi->flGraphicsCaps &= ~GCAPS_PANNING;

    /* SET GAMMA SUPPORT FOR PRIMARY SURFACE */

    if (ppdev->cPelSize != 0)
        pdi->flGraphicsCaps2 = GCAPS2_CHANGEGAMMARAMP;

	return ((DHPDEV) ppdev);

ReturnFailure1:

	DrvDisablePDEV((DHPDEV) ppdev);
	return 0;
}

/*-------------------------------------------------------------------------
 * DrvDisablePDEV
 *
 * Release the resources allocated in DrvEnablePDEV.  If a surface has been
 * enabled DrvDisableSurface will have already been called.
 *
 * Note that this function will be called when previewing modes in the
 * Display Applet, but not at system shutdown.  If you need to reset the
 * hardware at shutdown, you can do it in the miniport by providing a
 * 'HwResetHw' entry point in the VIDEO_HW_INITIALIZATION_DATA structure.
 *
 * Note: In an error, we may call this before DrvEnablePDEV is done.
 -------------------------------------------------------------------------*/

VOID DrvDisablePDEV (DHPDEV	dhpdev)
{
	PDEV	*ppdev = NULL;

	ppdev = (PDEV*) dhpdev;
	DISPDBG ((ppdev, 1, "DrvDisablePDEV\n"));

	/* FREE THE PALETTE AND PHYSICAL DEVICE MEMORY */

	vUninitializePalette(ppdev);
	EngFreeMem(ppdev);
}

/*-------------------------------------------------------------------------
 * DrvCompletePDEV
 *
 * Store the HPDEV, the engines handle for this PDEV, in the DHPDEV.
 -------------------------------------------------------------------------*/

VOID DrvCompletePDEV(
	DHPDEV dhpdev,
	HDEV   hdev)
{
	PDEV *ppdev = (PDEV *) dhpdev;

	DISPDBG ((ppdev, 1, "DrvCompletePDEV\n"));
	((PDEV*) dhpdev)->hdevEng = hdev;

	/* STORE THE HDEV GLOBALLY */
	/* This is a little bit of a gamble, as the thinking is that two */
	/* pdevs could exist simultaneously and simply be swapped back   */
	/* and forth.  With a single monitor system, this doesn't seem   */
	/* to be the case.                                               */

	global_hdevEng = hdev;

	return;
}

/*-------------------------------------------------------------------------
 * DrvEnableSurface
 *
 * Creates the drawing surface, initializes the hardware, and initializes
 * driver components.  This function is called after DrvEnablePDEV, and
 * performs the final device initialization.
 -------------------------------------------------------------------------*/

HSURF DrvEnableSurface (DHPDEV dhpdev)
{
	PDEV* ppdev = NULL;
	SIZEL sizl;
	SURFOBJ *pSurfObj;
	ULONG dwPitch;

	ppdev = (PDEV*) dhpdev;
	DISPDBG ((ppdev, 1, "DrvEnableSurface\n"));

	/* FIRST ENABLE ALL SUBCOMPONENTS                                        */
	/* Note that the order in which these 'Enable' functions are called      */
	/* may be significant in low off-screen memory conditions, because       */
	/* the off-screen heap manager may fail some of the later                */
	/* allocations...                                                        */

	if (!bEnableHardware(ppdev))
		goto ReturnFailure;

	if(!bEnablePalette(ppdev))
		goto ReturnFailure;

	if(!bEnableGamma(ppdev))
		goto ReturnFailure;

	sizl.cx = ppdev->cxScreen;
	sizl.cy = ppdev->cyScreen;

	/* REMEMBER THE SIZE OF THE SCREEN FOR CLIPPING */
	
	ppdev->scrnBounds.left   = 0;
	ppdev->scrnBounds.top    = 0;
	ppdev->scrnBounds.right  = ppdev->cxScreen;
	ppdev->scrnBounds.bottom = ppdev->cyScreen;

	ppdev->BytesPerPixel = (1 << ppdev->cPelSize);

	/* READ STRIDE */
		
	dwPitch = ppdev->myfuncs.pfn_get_display_pitch ();

	/* CREATE DEVICE BITMAP FOR FRAME BUFFER */

	ppdev->GXBitmap = EngCreateBitmap(
			sizl,			        /* Dimensions of bitmap               */
			dwPitch,	            /* Count of bytes to get to next line */
			ppdev->iBitmapFormat,   /* Bitmap format                      */
			BMF_TOPDOWN,	        /* Address bitmap the "right" way     */
			gfx_virt_fbptr);	    /* Pointer to data space              */

	/* CHECK LOW RESOURCE */

	if(!ppdev->GXBitmap) 
	{
		DISPDBG((ppdev, 3000, "DrvEnableSurface - Failed to create shadow bitmap\n"));
		goto ReturnFailure;
	}

	ppdev->hsurf = EngCreateDeviceSurface((DHSURF)1,
			sizl, ppdev->iBitmapFormat);

	if (!ppdev->hsurf)
	{
		DISPDBG((ppdev, 3000, "DrvEnableSurface - Failed EngCreateDeviceSurface\n"));
		goto ReturnFailure;
	}

    /* CREATE A FAKE SURFOBJ FOR DRIVER USE */

    pSurfObj = EngAllocMem(FL_ZERO_MEMORY, sizeof(SURFOBJ), gAllocTag);
	if (!pSurfObj)
    {
        DISPDBG((ppdev, 3000, "DrvEnableSurface - Failed EngAllocMem\n"));
		goto ReturnFailure;
    }

    pSurfObj->sizlBitmap.cx = ppdev->cxScreen;
	pSurfObj->dhsurf        = (DHSURF)1;
	pSurfObj->sizlBitmap.cy = ppdev->cyScreen;
    pSurfObj->dhpdev        = (DHPDEV)ppdev;
	pSurfObj->hdev          = ppdev->hdevEng;
	pSurfObj->pvScan0       = (void *) gfx_virt_fbptr;
	pSurfObj->lDelta        = dwPitch;
	pSurfObj->iUniq         = 0;
	pSurfObj->iBitmapFormat = ppdev->iBitmapFormat;	
	pSurfObj->iType         = STYPE_DEVICE;
	pSurfObj->fjBitmap      = 0;
    ppdev->pSurfObj         = pSurfObj; 	
    
    /* CREATE A LOCKED SURFACE FOR PUNTS TO GDI */

    ppdev->pLockedSurf = EngLockSurface ((HSURF)ppdev->GXBitmap);
    if (!ppdev->pLockedSurf)
    {
        DISPDBG((ppdev, 3000, "DrvEnableSurface - Failed EngLockSurface\n"));
		goto ReturnFailure;
    }

	DISPDBG((ppdev, 300, "Create surface 0x%x, mask 0x%x\n", (ULONG)ppdev->hsurf, ppdev->flHooks));

	if(!EngAssociateSurface(ppdev->hsurf, ppdev->hdevEng, ppdev->flHooks)) 
	{
		DISPDBG((ppdev, 300, "DrvEnableSurface - Failed associate for shadow bitmap\n"));
		goto ReturnFailure;
	}

	/* ZERO VIDEO OFFSET */
	/* This way if video gets read before it is properly initialized, */
	/* and we're running on a GXm that locks when you read off the    */
	/* end of video memory, we won't croak.                           */

	ppdev->myfuncs.pfn_set_video_offset (0);
	ppdev->OverlayActive  = FALSE;	
	ppdev->VisibleOverlay = 0;	

	DISPDBG((ppdev, 5, "Passed DrvEnableSurface\n"));
	return ppdev->hsurf;

ReturnFailure:

	DrvDisableSurface((DHPDEV) ppdev);
	return 0;
}

/*-------------------------------------------------------------------------
 * DrvDisableSurface
 *
 * Free resources allocated by DrvEnableSurface.  Release the surface.
 *
 * Note that this function will be called when previewing modes in the
 * Display Applet, but not at system shutdown.  If you need to reset the
 * hardware at shutdown, you can do it in the miniport by providing a
 * 'HwResetHw' entry point in the VIDEO_HW_INITIALIZATION_DATA structure.
 *
 * Note: In an error case, we may call this before DrvEnableSurface is
 *		completely done.
 -------------------------------------------------------------------------*/

VOID DrvDisableSurface (DHPDEV dhpdev)
{
	PDEV	*ppdev = NULL;

	ppdev = (PDEV*) dhpdev;

	/* SOME WORDS ABOUT DEALLOCATION */
	/* In an error case, some of the following relies on the    */
	/* fact that the PDEV is zero-initialized, so fields like   */
	/* 'hsurfScreen' will be zero unless the surface has been   */
	/* sucessfully initialized, and makes the assumption that   */
	/* EngDeleteSurface can take '0' as a parameter.            */

	vDisablePalette (ppdev);
	vDisableHardware (ppdev);
  
    if (ppdev->pLockedSurf)
        EngUnlockSurface (ppdev->pLockedSurf);
    if (ppdev->GXBitmap)
        EngDeleteSurface ((HSURF)(ppdev->GXBitmap));
    if (ppdev->pSurfObj)
        EngFreeMem (ppdev->pSurfObj);
    if (ppdev->hsurf)
	    EngDeleteSurface (ppdev->hsurf);
	
}

/*-------------------------------------------------------------------------
 * DrvAssertMode
 *
 * This asks the device to reset itself to the mode of the pdev passed in.
 -------------------------------------------------------------------------*/

BOOL DrvAssertMode(
	DHPDEV dhpdev,
	BOOL bEnable)
{
	PDEV* ppdev;
	ULONG data[2];
    ULONG ReturnedDataLength;

	ppdev = (PDEV*) dhpdev;

	DISPDBG((ppdev, 400, "DrvAssertMode pdev %x, enable %x\n",
		(ULONG)dhpdev, (ULONG) bEnable));

	if(!bEnable) 
	{
		/* DISABLE - SWITCH TO FULL-SCREEN */

		/* SET VGA MODE */
		/* Let's hope the mode-set back to graphics mode doesn't fail, */
		/* because we have no way of reporting an error back to GDI:   */

		bAssertModeHardware(ppdev, FALSE);

		// GIVE THE MINIPORT THE CURRENT HEAP INFORMATION
        // This routine is called before hibernating.  We take this opportunity to 
        // repack the heap and give the miniport a starting and ending position for the 
        // valid heap data.  In this way, the miniport can save only the data that 
        // it needs.  Note that this routine is also called when going into a full-screen
        // DOS mode.
        //
        RepackCache();

        // SAVE AS LITTLE DATA AS POSSIBLE
        // If no DirectDraw memory is being used, we only save and
        // restore the GDI heap.  Otherwise, we are forced to save
        // up to the end of the frame buffer.  Note that we never
        // save the active screen, assuming that it will be redrawn.
        //
        data[0] = heap_start;
        if (ddraw_offset == heap_end)
            data[1] = heap_offset;
        else
            data[1] = heap_end;
	
	    if(EngDeviceIoControl(ppdev->hDriver,
						    IOCTL_SET_HEAP_INFORMATION,
						    &data[0],  // input buffer 
						    8,
						    &data[0],  // output buffer 
						    8,
						    &ReturnedDataLength))
	    {
		    DISPDBG((ppdev, 3000, "DrvAssertMode - Failed IOCTL_SET_HEAP_INFORMATION\n"));
		    goto returnFalse;
	    }
	} 
	else 
	{
		/* ENABLE - SWITCH BACK TO GRAPHICS MODE */

		/* ENABLE ALL COMPONENTS */
		/* We have to enable every subcomponent in the reverse order    */
		/* in which it was disabled:                                    */

		bAssertModeHardware (ppdev, TRUE);
	}
	return TRUE;

returnFalse:

    return FALSE;
}

/*-------------------------------------------------------------------------
 * DrvGetModes
 *
 * Returns the list of available modes for the device.
 -------------------------------------------------------------------------*/

ULONG DrvGetModes(
	HANDLE hDriver,
	ULONG cjSize,
	DEVMODEW* pdm)
{
	DWORD cModes;
	DWORD cbOutputSize;
	PVIDEO_MODE_INFORMATION pVideoModeInformation;
	PVIDEO_MODE_INFORMATION pVideoTemp;
	DWORD cOutputModes = cjSize / (sizeof(DEVMODEW) + DRIVER_EXTRA_SIZE);
	DWORD cbModeSize;
	PDEV *ppdev = NULL;

	DISPDBG ((ppdev, 1, "DrvGetModes\n"));
	cModes = getAvailableModes(ppdev, hDriver,
							(PVIDEO_MODE_INFORMATION *) &pVideoModeInformation,
							&cbModeSize);
	if(cModes == 0) 
	{
		DISPDBG((ppdev, 300, "DrvGetModes failed to get mode information\n"));
		return 0;
	}

	if(pdm == NULL) 
	{
		cbOutputSize = cModes * (sizeof(DEVMODEW) + DRIVER_EXTRA_SIZE);
	} 
	else 
	{
		/* COPY THE INFORMATION FOR THE MODES BACK INTO THE OUTPUT BUFFER */

		cbOutputSize = 0;

		pVideoTemp = pVideoModeInformation;

		do 
		{
			if(pVideoTemp->Length != 0)
			{
				if (cOutputModes == 0) 
					break;
				
				/* ZERO THE STRUCTURE */

				memset(pdm, 0, sizeof(DEVMODEW));

				/* SET THE NAME OF THE DEVICE TO THE NAME OF THE DLL */

				memcpy(pdm->dmDeviceName, DLL_NAME, sizeof(DLL_NAME));

				pdm->dmSpecVersion   = DM_SPECVERSION;
				pdm->dmDriverVersion = DM_SPECVERSION;

				/* WE CURRENTLY DO NOT SUPPORT EXTRA INFORMATION IN THE DRIVER */

				pdm->dmDriverExtra = DRIVER_EXTRA_SIZE;

				pdm->dmSize             = sizeof(DEVMODEW);
				pdm->dmBitsPerPel       = pVideoTemp->NumberOfPlanes *
									      pVideoTemp->BitsPerPlane;
				pdm->dmPelsWidth        = pVideoTemp->VisScreenWidth;
				pdm->dmPelsHeight       = pVideoTemp->VisScreenHeight;
				pdm->dmDisplayFrequency = pVideoTemp->Frequency;

				pdm->dmFields = DM_BITSPERPEL | DM_PELSWIDTH |
						DM_PELSHEIGHT | DM_DISPLAYFREQUENCY | DM_DISPLAYFLAGS;

				/* ADVANCE TO NEXT DEVMODE ENTRY */

				cOutputModes--;

				pdm = (LPDEVMODEW) ( ((ULONG)pdm) + sizeof(DEVMODEW) +
												   DRIVER_EXTRA_SIZE);

				cbOutputSize += (sizeof(DEVMODEW) + DRIVER_EXTRA_SIZE);

			}

			pVideoTemp = (PVIDEO_MODE_INFORMATION)
				(((PUCHAR)pVideoTemp) + cbModeSize);

		} while(--cModes);
	}

	EngFreeMem(pVideoModeInformation);

	return cbOutputSize;
}

/*-------------------------------------------------------------------------
 * BOOL bAssertModeHardware
 *
 * Sets the appropriate hardware state for graphics mode or full-screen.
 -------------------------------------------------------------------------*/

BOOL bAssertModeHardware(
	PDEV* ppdev,
	BOOL bEnable)
{
	DWORD					    ReturnedDataLength = 0;
	ULONG					    ulReturn           = 0;
	GX_VIDEO_MODE_INFORMATION	VideoModeInfo;
	ULONG                       i, new_heap_start;
	
	if (bEnable) 
	{
		/* CALL THE MINIPORT TO SET THE GRAPHICS MODE */

		if(EngDeviceIoControl(ppdev->hDriver,
							 IOCTL_VIDEO_SET_CURRENT_MODE,
							 &ppdev->ulMode,  /* input buffer */
							 sizeof(DWORD),
							 NULL,
							 0,
							 &ReturnedDataLength))
		{
			DISPDBG((ppdev, 300, "bAssertModeHardware - Failed VIDEO_SET_CURRENT_MODE\n"));
			goto ReturnFalse;
		}
		
        /* RETRIEVE INFORMATION ABOUT THE MODE THAT WAS SET */

		if (EngDeviceIoControl(ppdev->hDriver,
							 IOCTL_VIDEO_QUERY_CURRENT_MODE,
							 NULL,
							 0,
							 &VideoModeInfo,
							 sizeof(VideoModeInfo),
							 &ReturnedDataLength))
		{
			DISPDBG((ppdev, 300, "bAssertModeHardware - failed VIDEO_QUERY_CURRENT_MODE\n"));
			goto ReturnFalse;
		}

		/* STORE MEMORY PARAMETERS */

		ppdev->lDelta	= VideoModeInfo.modeInfo.ScreenStride;
		ppdev->cxMemory = VideoModeInfo.modeInfo.VideoMemoryBitmapWidth;
		ppdev->cyMemory = VideoModeInfo.modeInfo.VideoMemoryBitmapHeight;
		
		/* CHECK MINIPORT VALUES */

		ASSERTVDD(ppdev->cxMemory >= ppdev->cxScreen, "Invalid cxMemory");
		ASSERTVDD(ppdev->cyMemory >= ppdev->cyScreen, "Invalid cyMemory");

		/* READ HEAP START */

		new_heap_start = VideoModeInfo.modeInfo.DriverSpecificAttributeFlags & 0xFFFFFF;
	
		compression_enabled = VideoModeInfo.modeInfo.DriverSpecificAttributeFlags & 0x40000000l;

		DISPDBG ((ppdev, 3000, "Heap Start is %8.8lX\n", new_heap_start));

		/* HEAP MANAGEMENT */
		/* The beginning of the offscreen heap varies based on the current mode. */
		/* The assumption we are making here is that the heap should be empty    */
		/* for each new mode.                                                    */

		heap_end = gfx_fb_size - 0x5000;

		/* INITIALIZE THE HEAP */
		/* This work is only done if we are setting a new resolution.  The other */
		/* cases involve sleep states or full screen DOS windows and do not      */
		/* require the heap to be set                                            */

		if (ppdev->cxScreen != previous_width || ppdev->cyScreen != previous_height ||
			ppdev->cBitsPerPel != previous_bpp)
		{
			if (previous_width == 0)
			{
				if (new_heap_start >= heap_end)
					heap_start = 0;
				
				else
				{
					heap_offset  = heap_start = new_heap_start;
					ddraw_offset = heap_end;
				}
			}
			
            EmptyBitmapHeap (ppdev, new_heap_start);
			ResetDDrawHeap();

            /* PRESET THE FIRST HEAP ENTRY */

            SetPrimaryHeapEntry (ppdev);
		}
		
		/* SAVE THE MODE PARAMETERS */

		previous_width  = ppdev->cxScreen;
		previous_height = ppdev->cyScreen;
		previous_bpp    = ppdev->cBitsPerPel;

        /* REMEMBER THE DISPLAY PITCH */

        ACQUIRE_PRIMITIVE_SEMAPHORE;
		mode_pitch = ppdev->myfuncs.pfn_get_display_pitch();
		mode_shift = (BYTE)(ppdev->cBitsPerPel + 7) >> 4;
		
		switch(mode_shift) 
		{
			case 0:
			default:
				ppdev->iBitmapFormat = BMF_8BPP;
				gu2_bpp = MGP_RM_BPPFMT_332;  
				break;

			case 1:
				ppdev->iBitmapFormat = BMF_16BPP;
				gu2_bpp = MGP_RM_BPPFMT_565;
				break;

			case 2:
				ppdev->iBitmapFormat = BMF_32BPP;
				gu2_bpp = MGP_RM_BPPFMT_8888;
				break;
		}

		/* ENABLE THE DISPLAY AND COMPRESSION */
		/* The display and compression are enabled in two places: during */
		/* a mode set when no bitmaps have yet to be deleted from the    */
		/* previous mode or immediately after the last bitmap is deleted */
		/* after a mode set.                                             */

		if (old_mode_count == 0)
		{
			ppdev->myfuncs.pfn_set_crt_enable(1);

			if (compression_enabled)
				ppdev->myfuncs.pfn_set_compression_enable(1);

			ppdev->myfuncs.pfn_set_raster_operation(0);
			ppdev->myfuncs.pfn_pattern_fill (0, 0, (unsigned short)ppdev->cxScreen, 
				(unsigned short)ppdev->cyScreen);
		}
		else
		{
			/* SAVE WIDTH AND HEIGHT TO BLANK THE SCREEN LATER */

			screen_width  = ppdev->cxScreen;
			screen_height = ppdev->cyScreen;
		}
        RELEASE_PRIMITIVE_SEMAPHORE;
	} 
	else 
	{
		/* RESET THE DEVICE TO A KNOWN STATE - MODE 3 */
		 
		if(EngDeviceIoControl(ppdev->hDriver,
							 IOCTL_VIDEO_RESET_DEVICE,
							 NULL,
							 0,
							 NULL,
							 0,
							 &ulReturn))
		{
			DISPDBG((ppdev, 300, "bAssertModeHardware - Failed reset IOCTL\n"));
			goto ReturnFalse;
		}
	}

	DISPDBG((ppdev, 300, "Passed bAssertModeHardware\n"));

	return(TRUE);

ReturnFalse:

	DISPDBG((ppdev, 300, "Failed bAssertModeHardware\n"));

	return(FALSE);
}

/*-------------------------------------------------------------------------
 * bEnableHardware
 *
 * Puts the hardware in the requested mode and initializes it.
 *
 * Note: Should be called before any access is done to the hardware from
 *		the display driver.
 -------------------------------------------------------------------------*/

BOOL bEnableHardware (PDEV *ppdev)
{
	VIDEO_PUBLIC_ACCESS_RANGES	VideoAccessRanges[7];
	VIDEO_MEMORY				VideoMemory;
	VIDEO_MEMORY_INFORMATION	VideoMemoryInfo;
	DWORD						ReturnedDataLength;
	UCHAR*						pj;
	USHORT* 					pw;
	ULONG*						pd;
	ULONG						i;
	ULONG						bb0Addr, bb1Addr;

	/* ACQUIRE MEMORY MAPPED POINTERS */
	/* The memory addresses are returned in the following order: */
	/*    0. Register Pointer                                    */
	/*    1. Video Pointer                                       */
	/*    2. GP Pointer                                          */
	/*    3. VGA Pointer                                         */
	/*    4. CPU Speed                                           */

	if(EngDeviceIoControl(ppdev->hDriver,
						 IOCTL_VIDEO_QUERY_PUBLIC_ACCESS_RANGES,
						 NULL,						/* input buffer */
						 0,
						 &VideoAccessRanges, 		/* output buffer */
						 sizeof(VideoAccessRanges),
						 &ReturnedDataLength))
	{
		DISPDBG((ppdev, 300, "bEnableHardware - Initialization error mapping IO port base\n"));
		goto ReturnFalse;
	}

	gfx_virt_regptr    = (unsigned char *) VideoAccessRanges[0].VirtualAddress;
	gfx_virt_vidptr    = (unsigned char *) VideoAccessRanges[1].VirtualAddress;
	gfx_virt_gpptr     = (unsigned char *) VideoAccessRanges[2].VirtualAddress;
	ppdev->CPUSpeed    = (USHORT) VideoAccessRanges[3].VirtualAddress;

	DISPDBG((ppdev, 100, "CPU speed returned   !!!!!!!!!!! %d\n", ppdev->CPUSpeed));

	/* GET THE LINEAR FRAME BUFFER ADDRESS */

	VideoMemory.RequestedVirtualAddress = NULL;

	if(EngDeviceIoControl(ppdev->hDriver,
						 IOCTL_VIDEO_MAP_VIDEO_MEMORY,
						 &VideoMemory,		/* input buffer */
						 sizeof(VIDEO_MEMORY),
						 &VideoMemoryInfo,	/* output buffer */
						 sizeof(VideoMemoryInfo),
						 &ReturnedDataLength))
	{
		DISPDBG((ppdev, 300, "bEnableHardware - Error mapping buffer address\n"));
		goto ReturnFalse;
	}

	/* RECORD THE LINEAR FRAME BUFFER ADDRESS */

	gfx_virt_fbptr = (unsigned char *) VideoMemoryInfo.FrameBufferBase;
	gfx_fb_size = VideoMemoryInfo.VideoRamLength;

	DISPDBG((ppdev, 300, "Register base 0x%X, Video base 0x%X, GP Base 0x%X, framebuffer base 0x%X\n",
		gfx_virt_regptr, gfx_virt_vidptr, gfx_virt_gpptr, gfx_virt_fbptr));
	DISPDBG((ppdev, 300, "Total FB size 0x%x\n", gfx_fb_size));

	/* SET THE MODE AND UNLOCK THE ACCELERATOR */
	
	if(!bAssertModeHardware(ppdev, TRUE))
		goto ReturnFalse;

	DISPDBG((ppdev, 100, "Passed bEnableHardware\n"));

	return(TRUE);

ReturnFalse:

	DISPDBG((ppdev, 300, "Failed bEnableHardware\n"));

	return(FALSE);
}

/*-------------------------------------------------------------------------
 * vDisableHardware
 *
 * Undoes anything done in bEnableHardware.
 *
 * Note: In an error case, we may call this before bEnableHardware is
 *	  	 completely done.
 -------------------------------------------------------------------------*/

VOID vDisableHardware (PDEV *ppdev)
{
	DWORD		 ReturnedDataLength;
	VIDEO_MEMORY VideoMemory[2];

	VideoMemory[0].RequestedVirtualAddress = gfx_virt_fbptr;

	if(EngDeviceIoControl(ppdev->hDriver,
						 IOCTL_VIDEO_UNMAP_VIDEO_MEMORY,
						 VideoMemory,
						 sizeof(VIDEO_MEMORY),
						 NULL,
						 0,
						 &ReturnedDataLength))
	{
		DISPDBG((ppdev, 300, "vDisableHardware failed IOCTL_VIDEO_UNMAP_VIDEO\n"));
	}

	VideoMemory[0].RequestedVirtualAddress = gfx_virt_regptr;
	
	if(EngDeviceIoControl(ppdev->hDriver,
						 IOCTL_VIDEO_FREE_PUBLIC_ACCESS_RANGES,
						 VideoMemory,
						 sizeof(VideoMemory),
						 NULL,
						 0,
						 &ReturnedDataLength))
	{
		DISPDBG((ppdev, 300, "vDisableHardware failed IOCTL_VIDEO_FREE_PUBLIC_ACCESS\n"));
	}
}

/*-------------------------------------------------------------------------
 * bInitializeModeFields
 *
 * Initializes a bunch of fields in the pdev, devcaps (aka gdiinfo), and
 * devinfo based on the requested mode.
 -------------------------------------------------------------------------*/

BOOL bInitializeModeFields(
	PDEV*	  ppdev,
	GDIINFO*  pgdi,
	DEVINFO*  pdi,
	DEVMODEW* pdm)
{
	ULONG					cModes;
	PVIDEO_MODE_INFORMATION pVideoBuffer;
	PVIDEO_MODE_INFORMATION pVideoModeSelected;
	PVIDEO_MODE_INFORMATION pVideoTemp;
	BOOL					bSelectDefault;
	VIDEO_MODE_INFORMATION	VideoModeInformation;
	ULONG					cbModeSize;

	/* CALL THE MINIPORT TO GET MODE INFORMATION */

	cModes = getAvailableModes(ppdev, ppdev->hDriver, &pVideoBuffer, &cbModeSize);
	if(cModes == 0)
	{
		DISPDBG((ppdev, 300, "Failed bInitializeModeFields\n"));
		return(FALSE);
	}

	/* ARE WE LOOKING FOR A DEFAULT MODE?                    */
	/* Width, height, BPP, flags and frequency will be zero. */

	if (((pdm->dmPelsWidth)	   ||
		 (pdm->dmPelsHeight)   ||
		 (pdm->dmBitsPerPel)   ||
		 (pdm->dmDisplayFlags) ||
		 (pdm->dmDisplayFrequency)) == 0)
	{
		bSelectDefault = TRUE;
	}
	else
	{
		bSelectDefault = FALSE;
	}

	/* SEARCH FOR A MATCH IN THE TABLE */

	pVideoModeSelected = NULL;
	pVideoTemp = pVideoBuffer;

	if (bSelectDefault)
	{
		DISPDBG((ppdev, 100, "Default mode requested\n"));
	}
	else
	{
		DISPDBG((ppdev, 100, "Requested mode...\n"));
		DISPDBG((ppdev, 100, "	Screen width  -- %li\n", pdm->dmPelsWidth));
		DISPDBG((ppdev, 100, "	Screen height -- %li\n", pdm->dmPelsHeight));
		DISPDBG((ppdev, 100, "	Bits per pel  -- %li\n", pdm->dmBitsPerPel));
		DISPDBG((ppdev, 100, "	Frequency	  -- %li\n", pdm->dmDisplayFrequency));
	}

	while(cModes--) 
	{
		/* CHECK THAT MODE HAS NOT BEEN REJECTED */

		if(pVideoTemp->Length != 0) 
		{
			DISPDBG((ppdev, 100, "	Checking against miniport mode:\n"));
			DISPDBG((ppdev, 100, "	   Screen width  -- %li\n", pVideoTemp->VisScreenWidth));
			DISPDBG((ppdev, 100, "	   Screen height -- %li\n", pVideoTemp->VisScreenHeight));
			DISPDBG((ppdev, 100, "	   Bits per pel  -- %li\n", pVideoTemp->BitsPerPlane *
													  pVideoTemp->NumberOfPlanes));
			DISPDBG((ppdev, 100, "	   Frequency	 -- %li\n", pVideoTemp->Frequency));

			if(bSelectDefault ||
				((pVideoTemp->VisScreenWidth  == pdm->dmPelsWidth)   &&
				 (pVideoTemp->VisScreenHeight == pdm->dmPelsHeight)  &&
				 (pVideoTemp->BitsPerPlane *
				  pVideoTemp->NumberOfPlanes  == pdm->dmBitsPerPel)) &&
				 (pVideoTemp->Frequency 	  == pdm->dmDisplayFrequency))
			{
				pVideoModeSelected = pVideoTemp;
				DISPDBG((ppdev, 100, "...Found a mode match!\n"));
				break;
			}
		}

		/* ADVANCE TO NEXT MODE */

		pVideoTemp = (PVIDEO_MODE_INFORMATION)
			(((PUCHAR)pVideoTemp) + cbModeSize);
	}

	/* DID WE FIND A MODE? */

	if(pVideoModeSelected == NULL)
	{
		DISPDBG((ppdev, 100, "Couldn't find a mode match!\n"));
		DISPDBG((ppdev, 300, "Failed bInitializeModeFields\n"));

		EngFreeMem(pVideoBuffer);		
		return(FALSE);
	}

	/* WE FOUND A MODE! */
	/* Save it in a stack buffer and get rid of allocated memory before */
	/* we forget to free it.                                            */

	VideoModeInformation = *pVideoModeSelected;
	EngFreeMem(pVideoBuffer);

	/* SET UP SCREEN INFORMATION FROM THE MINIPORT */

	ppdev->ulMode			= VideoModeInformation.ModeIndex;
	ppdev->cxScreen 		= VideoModeInformation.VisScreenWidth;
	ppdev->cyScreen 		= VideoModeInformation.VisScreenHeight;
	ppdev->cBitsPerPel		= VideoModeInformation.BitsPerPlane;
	ppdev->refreshRate		= VideoModeInformation.Frequency;

	DISPDBG((ppdev, 100, "ScreenStride: %lx\n", VideoModeInformation.ScreenStride));

	ppdev->flHooks = (HOOK_BITBLT
					| HOOK_COPYBITS
					| HOOK_PAINT
					| HOOK_STROKEPATH
                    | HOOK_LINETO
					| HOOK_TEXTOUT);

	/* HOOK ALPHA BLENDING FOR 32BPP */
	/* Windows does all alpha blending in 8:8:8:8 */

	if (ppdev->cBitsPerPel == 32)
		ppdev->flHooks |= HOOK_ALPHABLEND | HOOK_TRANSPARENTBLT;

	/* FILL IN THE GDIINFO STRUCTURE WITH THE DEFAULT 8BPP VALUES */

	*pgdi = ggdiDefault;

	/* NOW OVERWRITE THE DEFAULTS WITH THE RELEVANT INFORMATION RETURNED BY THE KERNEL DRIVER */

	pgdi->ulHorzSize		= VideoModeInformation.XMillimeter;
	pgdi->ulVertSize		= VideoModeInformation.YMillimeter;

	pgdi->ulHorzRes 		= VideoModeInformation.VisScreenWidth;
	pgdi->ulVertRes 		= VideoModeInformation.VisScreenHeight;
	pgdi->ulPanningHorzRes	= VideoModeInformation.VisScreenWidth;
	pgdi->ulPanningVertRes	= VideoModeInformation.VisScreenHeight;

	pgdi->cBitsPixel		= VideoModeInformation.BitsPerPlane;
	pgdi->cPlanes			= VideoModeInformation.NumberOfPlanes;
	pgdi->ulVRefresh		= VideoModeInformation.Frequency;

	pgdi->ulDACRed			= VideoModeInformation.NumberRedBits;
	pgdi->ulDACGreen		= VideoModeInformation.NumberGreenBits;
	pgdi->ulDACBlue 		= VideoModeInformation.NumberBlueBits;

	pgdi->ulLogPixelsX		= pdm->dmLogPixels;
	pgdi->ulLogPixelsY		= pdm->dmLogPixels;

	/* FILL IN THE DEVINFO STRUCTURE WITH THE DEFAULT 8BPP STRUCTURES */

	*pdi = gdevinfoDefault;

	if (VideoModeInformation.BitsPerPlane == 8)
	{
		ppdev->cPelSize 	   = 0;
		ppdev->iBitmapFormat   = BMF_8BPP;
		ppdev->ulWhite		   = 0xff;

		/* ASSUME ORTHOGONAL PALETTE - ALL COLORS ARE SAME SIZE */

		ppdev->cPaletteShift  = 8 - pgdi->ulDACRed;
		pdi->flGraphicsCaps  |= (GCAPS_PALMANAGED | GCAPS_COLOR_DITHER);
		DISPDBG((ppdev, 100, "palette shift = %d\n", ppdev->cPaletteShift));
	}
	else if ((VideoModeInformation.BitsPerPlane == 16) ||
			 (VideoModeInformation.BitsPerPlane == 15))
	{
		ppdev->cPelSize 	   = 1;
		ppdev->iBitmapFormat   = BMF_16BPP;
		ppdev->ulWhite		   = 0xffff;
		ppdev->flRed		   = VideoModeInformation.RedMask;
		ppdev->flGreen		   = VideoModeInformation.GreenMask;
		ppdev->flBlue		   = VideoModeInformation.BlueMask;

		pgdi->ulNumColors	   = (ULONG) -1;
		pgdi->ulNumPalReg	   = 0;
		pgdi->ulHTOutputFormat = HT_FORMAT_16BPP;

		pdi->iDitherFormat	   = BMF_16BPP;
	}
	else if (VideoModeInformation.BitsPerPlane == 32)
	{
		ppdev->cPelSize 	   = 2;
		ppdev->iBitmapFormat   = BMF_32BPP;
		ppdev->ulWhite		   = 0xffffff;
		ppdev->flRed		   = VideoModeInformation.RedMask;
		ppdev->flGreen		   = VideoModeInformation.GreenMask;
		ppdev->flBlue		   = VideoModeInformation.BlueMask;

		pgdi->ulNumColors	   = (ULONG) -1;
		pgdi->ulNumPalReg	   = 0;
		pgdi->ulHTOutputFormat = HT_FORMAT_32BPP;

		pdi->iDitherFormat	   = BMF_32BPP;
	}

	DISPDBG((ppdev, 100, "Passed bInitializeModeFields\n"));

	return(TRUE);	
}

/*-------------------------------------------------------------------------
 * getAvailableModes
 *
 * Calls the miniport to get the list of modes supported by the kernel driver,
 * and returns the list of modes supported by the display driver among those
 *
 * returns the number of entries in the videomode buffer.
 * 0 means no modes are supported by the miniport or that an error occured.
 *
 * NOTE: the buffer must be freed up by the caller.
 -------------------------------------------------------------------------*/

static DWORD getAvailableModes(
	PDEV					*ppdev,
	HANDLE					 hDriver,
	PVIDEO_MODE_INFORMATION* modeInformation,		/* Must be freed by caller */
	DWORD*					 cbModeSize)
{
	ULONG					ulTemp;
	VIDEO_NUM_MODES 		modes;
	PVIDEO_MODE_INFORMATION pVideoTemp;

	/* RETRIEVE THE NUMBER OF AVAILABLE MODES */

	if(EngDeviceIoControl(hDriver,
			IOCTL_VIDEO_QUERY_NUM_AVAIL_MODES,
			NULL,
			0,
			&modes,
			sizeof(VIDEO_NUM_MODES),
			&ulTemp))
	{
		DISPDBG((ppdev, 300, "getAvailableModes - Failed VIDEO_QUERY_NUM_AVAIL_MODES\n"));
		return(0);
	}

	*cbModeSize = modes.ModeInformationLength;

	/* ALLOCATE THE BUFFER TO HOLD THE VIDEO MODES */

	*modeInformation = (PVIDEO_MODE_INFORMATION)
						EngAllocMem(FL_ZERO_MEMORY,
									modes.NumModes *
									modes.ModeInformationLength,
									gAllocTag);

	/* CHECK LOW RESOURCE CASE */

	if (*modeInformation == (PVIDEO_MODE_INFORMATION) NULL)
	{
		DISPDBG((ppdev, 3000, "getAvailableModes - Failed EngAllocMem\n"));
		return 0;
	}

	/* CALL THE MINIPORT TO FILL IN ALL AVAILABLE MODES */

	if(EngDeviceIoControl(hDriver,
			IOCTL_VIDEO_QUERY_AVAIL_MODES,
			NULL,
			0,
			*modeInformation,
			modes.NumModes * modes.ModeInformationLength,
			&ulTemp))
	{
		DISPDBG((ppdev, 300, "getAvailableModes - Failed VIDEO_QUERY_AVAIL_MODES\n"));

		EngFreeMem(*modeInformation);
		*modeInformation = (PVIDEO_MODE_INFORMATION) NULL;

		return(0);
	}

	/* FILTER MODES THROUGH THE DISPLAY DRIVER */
	/* As an internal mechanism, set the length to 0 for the modes we */
	/* DO NOT support.                                                */

	ulTemp     =  modes.NumModes;
	pVideoTemp = *modeInformation;

	/* CHECK REJECT CASE                                                   */
	/* Mode is rejected if it is not one plane, or not graphics, or is not */
	/* one of 8, 15, 16 or 32BPP BPP.                                      */

	while(ulTemp--) 
	{
		if((pVideoTemp->NumberOfPlanes != 1 )                 ||
		  !(pVideoTemp->AttributeFlags & VIDEO_MODE_GRAPHICS) ||
		     ((pVideoTemp->BitsPerPlane != 8)  &&
		      (pVideoTemp->BitsPerPlane != 15) &&
			  (pVideoTemp->BitsPerPlane != 16) &&
	  	      (pVideoTemp->BitsPerPlane != 32))) 
		{

			DISPDBG((ppdev, 100, "Rejecting miniport mode:\n"));
			DISPDBG((ppdev, 100, "	Screen width  -- %li\n", pVideoTemp->VisScreenWidth));
			DISPDBG((ppdev, 100, "	Screen height -- %li\n", pVideoTemp->VisScreenHeight));
			DISPDBG((ppdev, 100, "	Bits per pel  -- %li\n", pVideoTemp->BitsPerPlane *
												   pVideoTemp->NumberOfPlanes));
			DISPDBG((ppdev, 100, "	Frequency	  -- %li\n", pVideoTemp->Frequency));

			pVideoTemp->Length = 0;
		}

		pVideoTemp = (PVIDEO_MODE_INFORMATION)
			(((PUCHAR)pVideoTemp) + modes.ModeInformationLength);
	}

	return(modes.NumModes);
}

